/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.notes.service.impl;

import java.io.*;

import com.google.common.io.Files;
import org.kuali.rice.kew.notes.Attachment;
import org.kuali.rice.kew.notes.service.AttachmentService;
import org.kuali.rice.kew.service.KEWServiceLocator;


/**
 * Implementation of the {@link AttachmentService}.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class AttachmentServiceImpl implements AttachmentService {
	
	protected final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(AttachmentServiceImpl.class);
	
	private static final String ATTACHMENT_PREPEND = "wf_att_";
	
	private String attachmentDir;

	@Override
    public void persistAttachedFileAndSetAttachmentBusinessObjectValue(Attachment attachment) throws Exception {
		createStorageDirIfNecessary();
		String uniqueId = KEWServiceLocator.getResponsibilityIdService().getNewResponsibilityId();
		String internalFileIndicator = attachment.getFileName().replace('.', '_');
		String fileName = ATTACHMENT_PREPEND + attachment.getNote().getDocumentId() + "_" + internalFileIndicator + "_" + uniqueId;
		File file = File.createTempFile(fileName, null, new File(attachmentDir));
		LOG.info("Persisting attachment at: " + file.getAbsolutePath());
		if (!file.canWrite()) {
			throw new RuntimeException("Do not have permissions to write to attachment file at: " + file.getAbsolutePath());
		}
		FileOutputStream streamOut = null;
        BufferedOutputStream bufferedStreamOut = null;
        try {
            streamOut = new FileOutputStream(file);
            bufferedStreamOut = new BufferedOutputStream(streamOut);
            int c;
            while ((c = attachment.getAttachedObject().read()) != -1) 
                {
                    bufferedStreamOut.write(c);
                }
        } finally {
        	if (bufferedStreamOut != null) {
        		bufferedStreamOut.close();
        	}
            if (streamOut != null) {
            	streamOut.close();
            }
        }
        attachment.setFileLoc(file.getAbsolutePath());
	}

	@Override
    public byte[] findAttachedFile(Attachment attachment) throws Exception {
		return Files.toByteArray(new File(attachment.getFileLoc()));
	}
	
	@Override
    public void deleteAttachedFile(Attachment attachment) throws Exception {
		File file = new File(attachment.getFileLoc());
		if (! file.delete()) {
			LOG.error("failed to delete file " + attachment.getFileLoc());
		}
	}
	
	private void createStorageDirIfNecessary() {
		if (attachmentDir == null) {
			throw new RuntimeException("Attachment Directory was not set when configuring workflow");
		}
		File attachDir = new File(attachmentDir);
		if (! attachDir.exists()) {
			LOG.warn("No attachment directory found.  Attempting to create directory " + attachmentDir);
			boolean directoriesCreated = attachDir.mkdirs();
			if (!directoriesCreated) {
				throw new RuntimeException("Failed to create directory for attachments at: " + attachDir.getAbsolutePath());
			}
		}
		if (!attachDir.canWrite()) {
			throw new RuntimeException("Do not have permission to write to: " + attachDir.getAbsolutePath());
		}
	}

	public String getAttachmentDir() {
		return attachmentDir;
	}

	public void setAttachmentDir(String attachmentDir) {
		this.attachmentDir = attachmentDir;
	}

}
