/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ken.web.spring;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.ken.bo.NotificationChannelBo;
import org.kuali.rice.ken.bo.UserChannelSubscriptionBo;
import org.kuali.rice.ken.service.NotificationChannelService;
import org.kuali.rice.ken.service.UserPreferenceService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.servlet.ModelAndView;

/**
 * This class is the controller that handles management of various user preferences interfaces (deliver types, user subscriptions, etc).
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@RequestMapping(value="/ken")
@Controller("userPreferencesController")
public class UserPreferencesController {
    
   private static String view = "";
   
   /** Logger for this class and subclasses */
   private static final Logger LOG = LogManager.getLogger(UserPreferencesController.class);

    @Autowired
    @Qualifier("notificationChannelService")
   protected NotificationChannelService notificationChannelService;

    @Autowired
    @Qualifier("userPreferenceService")
   protected UserPreferenceService userPreferenceService;

   /**
    * This method displays the actionList preferences screen.
    */
   @RequestMapping("/DisplayActionListPreferences.form")
   public ModelAndView displayActionListPreferences(HttpServletRequest request) {
       view = "ActionListPreferences";
       LOG.debug("remoteUser: "+request.getRemoteUser());
       Map<String, Object> model = new HashMap<String, Object>(); 
       return new ModelAndView(view, model);
   }
    
   /**
    * This method handles displaying the user preferences UI.
    */
   @RequestMapping("/DisplayUserPreferences.form")
   public ModelAndView displayUserPreferences(HttpServletRequest request) {

      view = "UserPreferencesForm";
      String userid = request.getRemoteUser();
      LOG.debug("remoteUser: "+userid);
      // get subscribable channels
      Collection<NotificationChannelBo> channels = this.notificationChannelService.getSubscribableChannels();
      // get current subscriptions for this user
      Collection<UserChannelSubscriptionBo> subscriptions = this.userPreferenceService.getCurrentSubscriptions(userid);
      Map<String, Object> currentsubs = new HashMap<String, Object>();
      Iterator<UserChannelSubscriptionBo> i = subscriptions.iterator();
      while (i.hasNext()) {
	  UserChannelSubscriptionBo sub = i.next();
	  String subid = Long.toString(sub.getChannel().getId());
 	  currentsubs.put(subid, subid);
	  LOG.debug("currently subscribed to: "+sub.getChannel().getId());
      }
      Map<String, Object> model = new HashMap<String, Object>();
      model.put("channels", channels);
      model.put("currentsubs", currentsubs);
      
      return new ModelAndView(view, model);
   }

   /**
    * Subscribe To a Channel
   */
   @RequestMapping("/SubscribeToChannel.form")
   public ModelAndView subscribeToChannel(HttpServletRequest request) {

       view = "UserPreferencesForm";
       String userid = request.getRemoteUser();
       LOG.debug("remoteUser: "+userid);
       String channelid = request.getParameter("channelid");
       NotificationChannelBo newChannel = this.notificationChannelService.getNotificationChannel(channelid);
       LOG.debug("newChannel name:"+newChannel.getName());
       UserChannelSubscriptionBo newSub = new UserChannelSubscriptionBo();
       newSub.setUserId(userid);
       newSub.setChannel(newChannel);
       LOG.debug("Calling service to subscribe to channel: "+newChannel.getName());
       this.userPreferenceService.subscribeToChannel(newSub);
       
       // get current subscription channel ids
       Collection<UserChannelSubscriptionBo> subscriptions = this.userPreferenceService.getCurrentSubscriptions(userid);
       Map<String, Object> currentsubs = new HashMap<String, Object>();;
       Iterator<UserChannelSubscriptionBo> i = subscriptions.iterator();
       while (i.hasNext()) {
 	  UserChannelSubscriptionBo sub = i.next();
 	  String subid = Long.toString(sub.getChannel().getId());
 	  currentsubs.put(subid, subid);
 	  LOG.debug("currently subscribed to: "+sub.getChannel().getId());
       }
       
       // get all subscribable channels       
       Collection<NotificationChannelBo> channels = this.notificationChannelService.getSubscribableChannels();
       
       Map<String, Object> model = new HashMap<String, Object>();
       model.put("channels", channels);
       model.put("currentsubs", currentsubs);
       return new ModelAndView(view, model);       
   }
   
   /**
    * Unsubscribe from Channel
    */
   @RequestMapping("/UnsubscribeFromChannel.form")
   public ModelAndView unsubscribeFromChannel(HttpServletRequest request) {
       view = "UserPreferencesForm";
       String userid = request.getRemoteUser();
       LOG.debug("remoteUser: "+userid);
       String channelid = request.getParameter("channelid");
       
       NotificationChannelBo newChannel = this.notificationChannelService.getNotificationChannel(channelid);
       LOG.debug("getting channel (id, user): "+channelid+","+userid); 
       UserChannelSubscriptionBo oldsub = this.userPreferenceService.getSubscription(channelid, userid);
       oldsub.setChannel(newChannel);
       
       LOG.debug("Calling service to unsubscribe: "+newChannel.getName());
       this.userPreferenceService.unsubscribeFromChannel(oldsub);
       LOG.debug("Finished unsubscribe service: "+newChannel.getName());
       
       // get current subscription channel ids
       Collection<UserChannelSubscriptionBo> subscriptions = this.userPreferenceService.getCurrentSubscriptions(userid);
       Map<String, Object> currentsubs = new HashMap<String, Object>();
       Iterator<UserChannelSubscriptionBo> i = subscriptions.iterator();
       while (i.hasNext()) {
 	  UserChannelSubscriptionBo sub = i.next();
 	  String subid = Long.toString(sub.getChannel().getId());
	  currentsubs.put(subid, subid);
 	  LOG.debug("currently subscribed to: "+sub.getChannel().getId());
       }
       
       // get all subscribable channels       
       Collection<NotificationChannelBo> channels = this.notificationChannelService.getSubscribableChannels();
       
       Map<String, Object> model = new HashMap<String, Object>();
       model.put("channels", channels);
       model.put("currentsubs", currentsubs);
       return new ModelAndView(view, model);    
        
   }

    public NotificationChannelService getNotificationChannelService() {
        return notificationChannelService;
    }

    public UserPreferenceService getUserPreferenceService() {
        return userPreferenceService;
    }

    public void setNotificationChannelService(NotificationChannelService notificationChannelService) {
        this.notificationChannelService = notificationChannelService;
    }

    public void setUserPreferenceService(UserPreferenceService userPreferenceService) {
        this.userPreferenceService = userPreferenceService;
    }

}
