/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.web;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.rice.kew.api.KewApiConstants;
import org.kuali.rice.kew.api.KewApiServiceLocator;
import org.kuali.rice.kew.api.preferences.Preferences;
import org.kuali.rice.kew.api.preferences.PreferencesService;
import org.kuali.rice.krad.UserSession;
import org.kuali.rice.krad.util.KRADUtils;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * This class establishes and initializes the KEW Preferences after a user logs in.
 * 
 * <p>
 * This filter assumes that a UserSession is already established.
 * </p>
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class UserPreferencesFilter implements Filter {
	
	private static final Logger LOG = LogManager.getLogger(UserPreferencesFilter.class);

	private FilterConfig filterConfig;
	private PreferencesService preferencesService;
	
	@Override
	public void init(FilterConfig config) throws ServletException {
		this.filterConfig = config;
	}

	@Override
	public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
		this.doFilter((HttpServletRequest) request, (HttpServletResponse) response, chain);
	}
	
	private void doFilter(HttpServletRequest request, HttpServletResponse response, FilterChain chain) throws IOException, ServletException {
		final UserSession session = KRADUtils.getUserSessionFromRequest(request);
		
		if (session == null) {
			throw new IllegalStateException("A user session has not been established");
		}
		
		final String principalId = session.getPrincipalId();
		
		if (session.retrieveObject(KewApiConstants.PREFERENCES) == null) {
			final Preferences preferences = retrievePreferences(principalId);
			session.addObject(KewApiConstants.PREFERENCES, preferences);
		}
		chain.doFilter(request, response);
	}

	@Override
	public void destroy() {
		filterConfig = null;
	}

	private Preferences retrievePreferences(String principalId) {
		synchronized (principalId) {
			Preferences preferences = this.getPreferenceService().getPreferences(principalId);
			if (preferences.isRequiresSave()) {
				LOG.info("Detected that user preferences require saving.");

				this.getPreferenceService().savePreferences(principalId, preferences);
				preferences = this.getPreferenceService().getPreferences(principalId);
			}

			return preferences;
		}
	}
    
    
    private PreferencesService getPreferenceService() {
    	if (this.preferencesService == null) {
    		this.preferencesService = KewApiServiceLocator.getPreferencesService();
    	}
    	
    	return this.preferencesService;
    }
    
}
