/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kim.impl.identity;

import org.kuali.rice.kim.api.identity.entity.EntityDefault;

/**
 * 
 * This service archives EntityDefault.  It's purpose is to provide long term 
 * storage for basic identity data that may be removed from the IdentityService implementation's
 * backing store.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 *
 */
public interface IdentityArchiveService {

    /**
     * Gets a {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} with an id from the archive.
     * {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} is a condensed version of {@link org.kuali.rice.kim.api.identity.entity.Entity} that contains
     * default values of its subclasses
     *
     * <p>
     *   This method will return null if the Entity does not exist.
     * </p>
     *
     * @param id the unique id to retrieve the entity by. cannot be null.
     * @return a {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} or null
     * @throws IllegalArgumentException if the id is blank
     */
    EntityDefault getEntityDefaultFromArchive(String id ) throws IllegalArgumentException;

	/**
     * Gets a {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} with an principalId from the archive.
     * {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} is a condensed version of {@link org.kuali.rice.kim.api.identity.entity.Entity} that contains
     * default values of its subclasses
     *
     * <p>
     *   This method will return null if the Entity does not exist.
     * </p>
     *
     * @param principalId the unique principalId to retrieve the entity by. cannot be null.
     * @return a {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} or null
     * @throws IllegalArgumentException if the principalId is blank
     */
    EntityDefault getEntityDefaultFromArchiveByPrincipalId(String principalId) throws IllegalArgumentException;

	/**
     * Gets a {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} with an principalName from the archive.
     * {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} is a condensed version of {@link org.kuali.rice.kim.api.identity.entity.Entity} that contains
     * default values of its subclasses
     *
     * <p>
     *   This method will return null if the Entity does not exist.
     * </p>
     *
     * @param principalName the unique principalName to retrieve the entity by. cannot be null.
     * @return a {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} or null
     * @throws IllegalArgumentException if the principalName is blank
     */
	EntityDefault getEntityDefaultFromArchiveByPrincipalName(String principalName) throws IllegalArgumentException;
	
    /**
     * Gets a {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} with an employeeId from the archive.
     * {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} is a condensed version of {@link org.kuali.rice.kim.api.identity.entity.Entity} that contains
     * default values of its subclasses
     *
     * <p>
     *   This method will return null if the Entity does not exist.
     * </p>
     *
     * @param employeeId the unique employeeId to retrieve the entity by. cannot be null.
     * @return a {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} or null
     * @throws IllegalArgumentException if the employeeId is blank
     */
    EntityDefault getEntityDefaultFromArchiveByEmployeeId(String employeeId) throws IllegalArgumentException;
    
	/**
     * Saves a {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} to the archive.
     * {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} is a condensed version of {@link org.kuali.rice.kim.api.identity.entity.Entity} that contains
     * default values of its subclasses
     *
     * <p>
     *   This method will return the saved EntityDefault object
     * </p>
     *
     * @param entityDefault the unique principalName to retrieve the entity by. cannot be null.
     * @return a {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} or null
     * @throws IllegalArgumentException if the entityDefault is null
     */
    //TODO: this should probably return some kind of Future<EntityDefault> if we can find a way to remote that
	void saveEntityDefaultToArchive(EntityDefault entityDefault) throws IllegalArgumentException;

    /**
     * Flushes {@link org.kuali.rice.kim.api.identity.entity.EntityDefault} to the archive.
     *
     * <p>
     *   This method flushes the "saved" entities to the database
     * </p>
     *
     * @return void
     */
    void flushToArchive() throws IllegalArgumentException;
	
}
