/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.dao.impl;

import org.apache.commons.lang.StringUtils;
import org.apache.ojb.broker.query.Criteria;
import org.apache.ojb.broker.query.QueryByCriteria;
import org.apache.ojb.broker.query.QueryFactory;
import org.kuali.rice.core.framework.persistence.ojb.dao.PlatformAwareDaoBaseOjb;
import org.kuali.rice.kns.service.KNSServiceLocator;
import org.kuali.rice.krad.bo.BusinessObject;
import org.kuali.rice.krad.bo.PersistableBusinessObject;
import org.kuali.rice.krad.dao.BusinessObjectDao;
import org.kuali.rice.krad.service.PersistenceStructureService;
import org.kuali.rice.krad.service.util.OjbCollectionAware;
import org.kuali.rice.krad.service.util.OjbCollectionHelper;
import org.kuali.rice.krad.util.KRADPropertyConstants;
import org.kuali.rice.krad.util.ObjectUtils;
import org.kuali.rice.krad.util.QueryPagingRequest;
import org.kuali.rice.krad.util.QueryPagingResults;
import org.springframework.dao.DataAccessException;
import org.springframework.orm.ObjectRetrievalFailureException;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * OJB implementation of the BusinessObjectDao interface and should be used for generic business object unit
 * tests
 */
@Deprecated
public class BusinessObjectDaoOjb extends PlatformAwareDaoBaseOjb implements BusinessObjectDao, OjbCollectionAware {
    private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(BusinessObjectDaoOjb.class);

    private PersistenceStructureService persistenceStructureService;
    private OjbCollectionHelper ojbCollectionHelper;

    public BusinessObjectDaoOjb(PersistenceStructureService persistenceStructureService) {
        this.persistenceStructureService = persistenceStructureService;
    }

	@Override
    public <T extends PersistableBusinessObject> T findBySinglePrimaryKey(Class<T> clazz, Object primaryKey) {
		if (primaryKey.getClass().getName().startsWith("java.lang.")
                || primaryKey.getClass().getName().startsWith("java.sql.")
                || primaryKey.getClass().getName().startsWith("java.math.")
                || primaryKey.getClass().getName().startsWith("java.util.")) {
			try {
				return (T) getPersistenceBrokerTemplate().getObjectById(clazz, primaryKey);
			} catch (  ObjectRetrievalFailureException ex  ) {
	    		// it doesn't exist, just return null
				return null;
			}
		} else {
			Criteria criteria = buildCriteria(clazz, primaryKey);

	        return (T) getPersistenceBrokerTemplate().getObjectByQuery(QueryFactory.newQuery(clazz, criteria));
		}
	}

    @Override
    public <T extends PersistableBusinessObject> T findByPrimaryKey(Class<T> clazz, Map<String, ?> primaryKeys) {
        final Criteria criteria = buildCriteria(primaryKeys);

        return (T) getPersistenceBrokerTemplate().getObjectByQuery(QueryFactory.newQuery(clazz, criteria));
    }

    /**
     * Retrieves all of the records for a given class name.
     *
     * @param clazz - the name of the object being used, either KualiCodeBase or a subclass
     * @return Collection
     */
    @Override
    public <T extends PersistableBusinessObject> List<T> findAll(Class<T> clazz) {
        return (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(QueryFactory.newQuery(clazz, (Criteria) null));
    }

    @Override
    public <T extends PersistableBusinessObject> QueryPagingResults<T> findAll(Class<T> clazz, QueryPagingRequest pagingRequest) {
        final int total = getPersistenceBrokerTemplate().getCount(QueryFactory.newQuery(clazz, (Criteria) null));
        final List<T> results = (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(withPaging(QueryFactory.newQuery(clazz, (Criteria) null), pagingRequest));
        return new QueryPagingResults<>(pagingRequest, results, total);
    }

    private QueryByCriteria withPaging(QueryByCriteria queryByCriteria, QueryPagingRequest pagingRequest) {
        final int ojbStartIndex = pagingRequest.getStartAtIndex() + 1;
        final int ojbEndIndex = ojbStartIndex + pagingRequest.getMaxResults();
        queryByCriteria.setStartAtIndex(ojbStartIndex);
        queryByCriteria.setEndAtIndex(ojbEndIndex);
        return queryByCriteria;
    }

    @Override
    public <T extends PersistableBusinessObject> List<T> findAllOrderBy(Class<T> clazz, String sortField, boolean sortAscending) {
        final QueryByCriteria queryByCriteria = new QueryByCriteria(clazz, null);

        if (sortAscending) {
            queryByCriteria.addOrderByAscending(sortField);
        } else {
            queryByCriteria.addOrderByDescending(sortField);
        }

        return (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(queryByCriteria);
    }

    @Override
    public <T extends PersistableBusinessObject> QueryPagingResults<T> findAllOrderBy(Class<T> clazz, String sortField, boolean sortAscending, QueryPagingRequest pagingRequest) {
        final QueryByCriteria queryByCriteria = new QueryByCriteria(clazz, null);

        if (sortAscending) {
            queryByCriteria.addOrderByAscending(sortField);
        } else {
            queryByCriteria.addOrderByDescending(sortField);
        }

        final int total = getPersistenceBrokerTemplate().getCount(queryByCriteria);
        final List<T> results = (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(withPaging(queryByCriteria, pagingRequest));
        return new QueryPagingResults<>(pagingRequest, results, total);
    }

    /**
     * This is the default impl that comes with Kuali - uses OJB.
     */
    @Override
    public <T extends PersistableBusinessObject> List<T> findMatching(Class<T> clazz, Map<String, ?> fieldValues) {
        final Criteria criteria = buildCriteria(fieldValues);

        return (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(QueryFactory.newQuery(clazz, criteria));
    }

    @Override
    public <T extends PersistableBusinessObject> QueryPagingResults<T> findMatching(Class<T> clazz, Map<String, ?> fieldValues, QueryPagingRequest pagingRequest) {
        final Criteria criteria = buildCriteria(fieldValues);
        final QueryByCriteria queryByCriteria = QueryFactory.newQuery(clazz, criteria);
        final int total = getPersistenceBrokerTemplate().getCount(queryByCriteria);
        final List<T> results = (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(withPaging(queryByCriteria, pagingRequest));
        return new QueryPagingResults<>(pagingRequest, results, total);
    }

    @Override
    public <T extends PersistableBusinessObject> List<T> findAllActive(Class<T> clazz) {
        return (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(QueryFactory.newQuery(clazz, buildActiveCriteria()));
    }

    @Override
    public <T extends PersistableBusinessObject> QueryPagingResults<T> findAllActive(Class<T> clazz, QueryPagingRequest pagingRequest) {
        final Criteria criteria = buildActiveCriteria();
        final QueryByCriteria queryByCriteria = QueryFactory.newQuery(clazz, criteria);
        final int total = getPersistenceBrokerTemplate().getCount(queryByCriteria);
        final List<T> results = (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(withPaging(queryByCriteria, pagingRequest));
        return new QueryPagingResults<>(pagingRequest, results, total);
    }

    @Override
    public <T extends PersistableBusinessObject> List<T> findAllInactive(Class<T> clazz) {
        return (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(QueryFactory.newQuery(clazz, buildInactiveCriteria()));
    }

    @Override
    public <T extends PersistableBusinessObject> QueryPagingResults<T> findAllInactive(Class<T> clazz, QueryPagingRequest pagingRequest) {
        final Criteria criteria = buildInactiveCriteria();
        final QueryByCriteria queryByCriteria = QueryFactory.newQuery(clazz, criteria);
        final int total = getPersistenceBrokerTemplate().getCount(queryByCriteria);
        final List<T> results = (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(withPaging(queryByCriteria, pagingRequest));
        return new QueryPagingResults<>(pagingRequest, results, total);
    }

    @Override
    public <T extends PersistableBusinessObject> List<T> findAllActiveOrderBy(Class<T> clazz, String sortField, boolean sortAscending) {
        final QueryByCriteria queryByCriteria = new QueryByCriteria(clazz, buildActiveCriteria());

        if (sortAscending) {
            queryByCriteria.addOrderByAscending(sortField);
        } else {
            queryByCriteria.addOrderByDescending(sortField);
        }

        return (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(queryByCriteria);
    }

    @Override
    public <T extends PersistableBusinessObject> QueryPagingResults<T> findAllActiveOrderBy(Class<T> clazz, String sortField, boolean sortAscending, QueryPagingRequest pagingRequest) {
        final QueryByCriteria queryByCriteria = new QueryByCriteria(clazz, buildActiveCriteria());

        if (sortAscending) {
            queryByCriteria.addOrderByAscending(sortField);
        } else {
            queryByCriteria.addOrderByDescending(sortField);
        }

        final int total = getPersistenceBrokerTemplate().getCount(queryByCriteria);
        final List<T> results = (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(withPaging(queryByCriteria, pagingRequest));
        return new QueryPagingResults<>(pagingRequest, results, total);
    }

    @Override
    public <T extends PersistableBusinessObject> List<T> findMatchingActive(Class<T> clazz, Map<String, ?> fieldValues) {
        final Criteria criteria = buildCriteria(fieldValues);
        criteria.addAndCriteria(buildActiveCriteria());

        return (List<T>)getPersistenceBrokerTemplate().getCollectionByQuery(QueryFactory.newQuery(clazz, criteria));
    }

    @Override
    public <T extends PersistableBusinessObject> QueryPagingResults<T> findMatchingActive(Class<T> clazz, Map<String, ?> fieldValues, QueryPagingRequest pagingRequest) {
        final Criteria criteria = buildCriteria(fieldValues);
        criteria.addAndCriteria(buildActiveCriteria());
        final QueryByCriteria queryByCriteria = QueryFactory.newQuery(clazz, criteria);

        final int total = getPersistenceBrokerTemplate().getCount(queryByCriteria);
        final List<T> results = (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(withPaging(queryByCriteria, pagingRequest));
        return new QueryPagingResults<>(pagingRequest, results, total);
    }

    /**
     * This is the default impl that comes with Kuali - uses OJB.
     */
    @Override
    public <T extends PersistableBusinessObject> int countMatching(Class<T> clazz, Map<String, ?> fieldValues) {
        final Criteria criteria = buildCriteria(fieldValues);

        return getPersistenceBrokerTemplate().getCount(QueryFactory.newQuery(clazz, criteria));
    }

    /**
     * This is the default impl that comes with Kuali - uses OJB.
     */
    @Override
    public <T extends PersistableBusinessObject> int countMatching(Class<T> clazz, Map<String, ?> positiveFieldValues, Map<String, ?> negativeFieldValues) {
        final Criteria criteria = buildCriteria(positiveFieldValues);
        final Criteria negativeCriteria = buildNegativeCriteria(negativeFieldValues);
        criteria.addAndCriteria(negativeCriteria);
        return getPersistenceBrokerTemplate().getCount(QueryFactory.newQuery(clazz, criteria));
    }

    /**
     * This is the default impl that comes with Kuali - uses OJB.
     */
    @Override
    public <T extends PersistableBusinessObject> List<T> findMatchingOrderBy(Class<T> clazz, Map<String, ?> fieldValues, String sortField, boolean sortAscending) {
        final Criteria criteria = buildCriteria(fieldValues);
        final QueryByCriteria queryByCriteria = new QueryByCriteria(clazz, criteria);

        if (sortAscending) {
            queryByCriteria.addOrderByAscending(sortField);
        } else {
            queryByCriteria.addOrderByDescending(sortField);
        }

        return (List<T>)getPersistenceBrokerTemplate().getCollectionByQuery(queryByCriteria);
    }

    @Override
    public <T extends PersistableBusinessObject> QueryPagingResults<T> findMatchingOrderBy(Class<T> clazz, Map<String, ?> fieldValues, String sortField, boolean sortAscending, QueryPagingRequest pagingRequest) {
        final Criteria criteria = buildCriteria(fieldValues);
        final QueryByCriteria queryByCriteria = new QueryByCriteria(clazz, criteria);

        if (sortAscending) {
            queryByCriteria.addOrderByAscending(sortField);
        } else {
            queryByCriteria.addOrderByDescending(sortField);
        }

        final int total = getPersistenceBrokerTemplate().getCount(queryByCriteria);
        final List<T> results = (List<T>) getPersistenceBrokerTemplate().getCollectionByQuery(withPaging(queryByCriteria, pagingRequest));
        return new QueryPagingResults<>(pagingRequest, results, total);
    }

    /**
	 * Saves a business object.
	 */
	@Override
    public <T extends PersistableBusinessObject> T save(T bo) throws DataAccessException {
		// if collections exist on the BO, create a copy and use to process the
		// collections to ensure
		// that removed elements are deleted from the database
		Set<String> boCollections = getPersistenceStructureService().listCollectionObjectTypes(bo.getClass()).keySet();
		PersistableBusinessObject savedBo;
		if (!boCollections.isEmpty()) {
			// refresh bo to get db copy of collections
			savedBo = (PersistableBusinessObject) ObjectUtils.deepCopy(bo);
			for (String boCollection : boCollections) {
				if (getPersistenceStructureService().isCollectionUpdatable(savedBo.getClass(), boCollection)) {
					savedBo.refreshReferenceObject(boCollection);
				}
			}
            getOjbCollectionHelper().processCollections(this, bo, savedBo);
        }

		getPersistenceBrokerTemplate().store(bo);
		return bo;
	}

    /**
     * Saves a business object.
     */
    @Override
    public <T extends PersistableBusinessObject> List<T> save(List<T> businessObjects) throws DataAccessException {
    	if ( LOG.isDebugEnabled() ) {
    		LOG.debug( "About to persist the following BOs:" );
            businessObjects.forEach(bo -> LOG.debug( "   --->" + bo));
    	}
        businessObjects.forEach(bo -> getPersistenceBrokerTemplate().store(bo));
        return businessObjects;
    }


    /**
     * Deletes the business object passed in.
     */
    @Override
    public void delete(Object bo) {
        getPersistenceBrokerTemplate().delete(bo);
    }

    @Override
    public <T extends PersistableBusinessObject> void delete(List<T> boList) {
        for (PersistableBusinessObject bo : boList) {
            getPersistenceBrokerTemplate().delete(bo);
        }
    }

    @Override
    public <T extends PersistableBusinessObject> void deleteMatching(Class<T> clazz, Map<String, ?> fieldValues) {
        Criteria criteria = buildCriteria(fieldValues);

        getPersistenceBrokerTemplate().deleteByQuery(QueryFactory.newQuery(clazz, criteria));

        // An ojb delete by query doesn't update the cache so we need to clear the cache for everything to work property.
        // don't believe me? Read the source code to OJB
        getPersistenceBrokerTemplate().clearCache();
    }

    @Override
    public Object retrieve(Object object) {
        return getPersistenceBrokerTemplate().getObjectByQuery(QueryFactory.newQueryByIdentity(object));
    }

    /**
	 * OJB does not support this method
	 */
	@Override
    public <T extends PersistableBusinessObject> T findByPrimaryKeyUsingKeyObject(Class<T> clazz, Object pkObject) {
		throw new UnsupportedOperationException("OJB does not support this option");
	}

	/**
	 * No need to do anything - avoid saving and OJB will "manage read only"
	 */
	@Override
    public <T extends PersistableBusinessObject> T manageReadOnly(T bo) {
		return bo;
	}

	/**
     * This method will build out criteria in the key-value paradigm (attribute-value).
     */
    private Criteria buildCriteria(Map<String, ?> fieldValues) {
        Criteria criteria = new Criteria();
        for (Map.Entry<String, ?> stringEntry : fieldValues.entrySet()) {
            String key = stringEntry.getKey();
            Object value = stringEntry.getValue();
            if (value instanceof Collection) {
                criteria.addIn(key, (Collection) value);
            } else {
                criteria.addEqualTo(key, value);
            }
        }

        return criteria;
    }

    
    private <T extends BusinessObject> Criteria buildCriteria(Class<T> clazz, Object primaryKey) {
        Map<String, Object> fieldValues = new HashMap<>();
        List<String> fieldNames = getPersistenceStructureService().getPrimaryKeys(clazz);

        //create map of values
        for (String fieldName : fieldNames) {
            Object fieldValue;

            try {
                fieldValue = primaryKey.getClass().getMethod("get" + StringUtils.capitalize(fieldName)).invoke(primaryKey);
                fieldValues.put(fieldName, fieldValue);
            } catch (IllegalArgumentException | IllegalAccessException | SecurityException | InvocationTargetException | NoSuchMethodException e) {
                LOG.error(e.getMessage(), e);
            }
        }
        return this.buildCriteria(fieldValues);
    }
    
    /**
     * Builds a Criteria object for active field set to true
     * @return Criteria
     */
    private Criteria buildActiveCriteria(){
        Criteria criteria = new Criteria();
        criteria.addEqualTo(KRADPropertyConstants.ACTIVE, true);

        return criteria;
    }

    /**
     * Builds a Criteria object for active field set to true
     * @return Criteria
     */
    private Criteria buildInactiveCriteria(){
        Criteria criteria = new Criteria();
        criteria.addEqualTo(KRADPropertyConstants.ACTIVE, false);

        return criteria;
    }

    /**
     * This method will build out criteria in the key-value paradigm (attribute-value).
     */
    private Criteria buildNegativeCriteria(Map<String, ?> negativeFieldValues) {
        Criteria criteria = new Criteria();
        for (Map.Entry<String, ?> stringEntry : negativeFieldValues.entrySet()) {
            String key = stringEntry.getKey();
            Object value = stringEntry.getValue();
            if (value instanceof Collection) {
                criteria.addNotIn(key, (Collection) value);
            } else {
                criteria.addNotEqualTo(key, value);
            }
        }

        return criteria;
    }

    protected PersistenceStructureService getPersistenceStructureService() {
        return persistenceStructureService;
    }

    public void setPersistenceStructureService(PersistenceStructureService persistenceStructureService) {
        this.persistenceStructureService = persistenceStructureService;
    }

    protected OjbCollectionHelper getOjbCollectionHelper() {
        if (ojbCollectionHelper == null) {
            ojbCollectionHelper = KNSServiceLocator.getOjbCollectionHelper();
        }

        return ojbCollectionHelper;
    }

    public void setOjbCollectionHelper(OjbCollectionHelper ojbCollectionHelper) {
        this.ojbCollectionHelper = ojbCollectionHelper;
    }
}
