/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.bo;

import org.joda.time.DateTime;
import org.kuali.rice.core.api.mo.common.active.InactivatableFromToUtils;

import javax.persistence.Column;
import javax.persistence.Embeddable;
import javax.persistence.MappedSuperclass;
import javax.persistence.Transient;
import java.sql.Timestamp;

/**
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@MappedSuperclass
public abstract class InactivatableFromToImpl extends PersistableBusinessObjectBase implements InactivatableFromTo {

	private static final long serialVersionUID = 1L;

    /**
     * EclipseLink static weaving does not weave MappedSuperclass unless an Entity or Embedded is
     * weaved which uses it, hence this class.
     */
    @Embeddable
    private static final class WeaveMe extends InactivatableFromToImpl {}

    @Column(name = "ACTV_FRM_DT")
	protected Timestamp activeFromDate;
	@Column(name = "ACTV_TO_DT")
	protected Timestamp activeToDate;
	@Transient
	protected Timestamp activeAsOfDate;
	@Transient
	protected boolean current;

	/**
	 * Returns active if the {@link #getActiveAsOfDate()} (current time used if not set) is between
	 * the from and to dates. Null dates are considered to indicate an open range.
	 */
	@Override
    public boolean isActive() {
        return InactivatableFromToUtils.isActive(activeFromDate == null ? null : new DateTime(activeFromDate),
                                                 activeToDate == null ? null : new DateTime(activeToDate),
                                                 new DateTime(activeAsOfDate));
	}
	
	@Override
    public void setActive(boolean active) {
		// do nothing
	}

	@Override
    public void setActiveFromDate(Timestamp from) {
		this.activeFromDate = from;
	}

	@Override
    public void setActiveToDate(Timestamp to) {
		this.activeToDate = to;
	}

	@Override
    public Timestamp getActiveFromDate() {
		return this.activeFromDate;
	}

	@Override
    public Timestamp getActiveToDate() {
		return this.activeToDate;
	}

	@Override
    public Timestamp getActiveAsOfDate() {
		return this.activeAsOfDate;
	}

	@Override
    public void setActiveAsOfDate(Timestamp activeAsOfDate) {
		this.activeAsOfDate = activeAsOfDate;
	}

	public boolean isCurrent() {
		return this.current;
	}

	public void setCurrent(boolean current) {
		this.current = current;
	}
}
