/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kns.datadictionary.control;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang.ClassUtils;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.kuali.rice.core.api.util.ClassLoaderUtils;
import org.kuali.rice.krad.bo.BusinessObject;
import org.kuali.rice.krad.datadictionary.DataDictionaryDefinitionBase;
import org.kuali.rice.krad.datadictionary.control.ControlDefinition;
import org.kuali.rice.krad.datadictionary.exception.ClassValidationException;
import org.kuali.rice.krad.datadictionary.exception.CompletionException;
import org.kuali.rice.krad.keyvalues.KeyValuesFinder;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * A single HTML control definition in the DataDictionary, which contains information relating to the HTML control used to realize a
 * specific attribute. All types of controls are represented by an instance of this class; you have to call one of the is* methods
 * to figure out which of the other accessors should return useful values.
 *
 * @deprecated Only used by KNS classes, use KRAD.
 */
@Deprecated
public abstract class ControlDefinitionBase extends DataDictionaryDefinitionBase implements ControlDefinition {
    private static final long serialVersionUID = 4372435175782501152L;
    private static final String BUSINESS_OBJECT_CLASS = "businessObjectClass";
    private static final String INCLUDE_BLANK_ROW = "includeBlankRow";
    private static final String INCLUDE_KEY_IN_LABEL = "includeKeyInLabel";
    private static final String KEY_ATTRIBUTE = "keyAttribute";
    private static final String LABEL_ATTRIBUTE = "labelAttribute";

    protected boolean datePicker;
	protected String datePickerFormat;
	protected boolean expandedTextArea;
    protected String script;
    protected String valuesFinderClass;
    protected Map<String, Object> valuesFinderProperties = new HashMap<>();
    protected Integer size;
    protected Integer rows;
    protected Integer cols;
    protected boolean ranged;


    public ControlDefinitionBase() {
    	ranged = true;
    }

    @Override
    public boolean isDatePicker() {
        return datePicker;
    }

    /** Whether this control should have a date picker button next to the field.
     *  Valid for text fields.
     */
    @Override
    public void setDatePicker(boolean datePicker) {
        this.datePicker=datePicker;
    }

    @Override
    public String getDatePickerFormat() {
        return datePickerFormat;
    }

    @Override
    public void setDatePickerFormat(String datePickerFormat) {
        this.datePickerFormat = datePickerFormat;
    }

    @Override
    public boolean isExpandedTextArea() {
        return expandedTextArea;
    }

    /** Whether this control should have a expanded text area button next to the field.
     *  Valid for textarea fields.
     */
    @Override
    public void setExpandedTextArea(boolean eTextArea) {
        this.expandedTextArea=eTextArea;
    }


    @Override
    public boolean isCheckbox() {
        return false;
    }


    @Override
    public boolean isHidden() {
        return false;
    }


    @Override
    public boolean isRadio() {
        return false;
    }


    @Override
    public boolean isSelect() {
        return false;
    }
    

    @Override
    public boolean isMultiselect() {
        return false;
    }


    @Override
    public boolean isText() {
        return false;
    }


    @Override
    public boolean isTextarea() {
        return false;
    }


    @Override
    public boolean isCurrency() {
        return false;
    }


    @Override
    public boolean isKualiUser() {
        return false;
    }

    @Override
    public boolean isWorkflowWorkgroup() {
        return false;
    }


    @Override
    public boolean isFile() {
        return false;
    }


    @Override
    public boolean isLookupHidden() {
        return false;
    }


    @Override
    public boolean isLookupReadonly() {
        return false;
    }
    

    @Override
    public boolean isButton() {
        return false;
    }
    

    @Override
    public boolean isLink() {
        return false;
    }
    

    @Override
    public void setValuesFinderClass(String valuesFinderClass) {
        if (valuesFinderClass == null) {
            throw new IllegalArgumentException("invalid (null) valuesFinderClass");
        }

        this.valuesFinderClass = valuesFinderClass;
    }

    @Override
    public void setValuesFinderProperties(Map<String, Object> valuesFinderProperties) {
        this.valuesFinderProperties = valuesFinderProperties;
    }

    /**
     * @return the dataObjectClass
     */
    @Override
    public String getBusinessObjectClass() {
        final Object v = valuesFinderProperties.get(BUSINESS_OBJECT_CLASS);
        return v != null ? v.toString() : null;
    }

    /**
     * Used by a PersistableBusinessObjectValuesFinder to automatically query and display a list
     * of business objects as part of a select list or set of radio buttons.
     * 
     * The keyAttribute, labelAttribute, and includeKeyInLabel are used with this property.
     * 
     * @param businessObjectClass the dataObjectClass to set
     */
    @Override
    public void setBusinessObjectClass(String businessObjectClass) {
        if (businessObjectClass == null) {
            throw new IllegalArgumentException("invalid (null) dataObjectClass");
        }

        valuesFinderProperties.put(BUSINESS_OBJECT_CLASS, businessObjectClass);
    }

    /**
     * @return the includeBlankRow
     */
    @Override
    public Boolean getIncludeBlankRow() {
        final Object v = valuesFinderProperties.get(INCLUDE_BLANK_ROW);
        return v != null && Boolean.parseBoolean(v.toString());
    }

    @Override
    public void setIncludeBlankRow(Boolean includeBlankRow) {
        valuesFinderProperties.put(INCLUDE_BLANK_ROW, includeBlankRow.toString());
    }

    /**
     * @return the includeKeyInLabel
     */
    @Override
    public Boolean getIncludeKeyInLabel() {
        final Object v = valuesFinderProperties.get(INCLUDE_KEY_IN_LABEL);
        return v != null && Boolean.parseBoolean(v.toString());
    }

    /**
     * Whether to include the key in the label for select lists and radio buttons.
     */
    @Override
    public void setIncludeKeyInLabel(Boolean includeKeyInLabel) {
        valuesFinderProperties.put(INCLUDE_KEY_IN_LABEL, includeKeyInLabel.toString());
    }

    /**
     * @return the keyAttribute
     */
    @Override
    public String getKeyAttribute() {
        final Object v = valuesFinderProperties.get(KEY_ATTRIBUTE);
        return v != null ? v.toString() : null;
    }

    /**
     * Attribute of the given dataObjectClass to use as the value of a select list
     * or set of radio buttons. 
     */
    @Override
    public void setKeyAttribute(String keyAttribute) {
        valuesFinderProperties.put(KEY_ATTRIBUTE, keyAttribute);
    }

    /**
     * @return the labelAttribute
     */
    @Override
    public String getLabelAttribute() {
        final Object v = valuesFinderProperties.get(LABEL_ATTRIBUTE);
        return v != null ? v.toString() : null;
    }

    /**
     * Attribute of the given dataObjectClass to use as the displayed label on a select list
     * or set of radio buttons. 
     */
    @Override
    public void setLabelAttribute(String labelAttribute) {
        valuesFinderProperties.put(LABEL_ATTRIBUTE, labelAttribute);
    }

    @Override
    public String getValuesFinderClass() {
        return valuesFinderClass;
    }

    @Override
    public Map<String, Object> getValuesFinderProperties() {
        return valuesFinderProperties;
    }

    /**
     * Size of a text control.
     * 
     */
    @Override
    public void setSize(Integer size) {
        this.size = size;
    }


    @Override
    public Integer getSize() {
        return size;
    }


    public boolean hasScript() {
        return false;
    }

    /**
     * Number of rows to display on a text-area widget.
     */
    @Override
    public void setRows(Integer rows) {
        this.rows = rows;
    }

    @Override
    public Integer getRows() {
        return rows;
    }

    /**
     * Number of columns to display on a text-area widget.
     */
    @Override
    public void setCols(Integer cols) {
        this.cols = cols;
    }

    @Override
    public Integer getCols() {
        return cols;
    }

    /**
     * Directly validate simple fields
     */
    @Override
    public void completeValidation(Class<?> rootBusinessObjectClass, Class<?> otherBusinessObjectClass) {
        if (!isCheckbox() && !isHidden() && !isRadio() && !isSelect() && !isMultiselect() && !isText() && !isTextarea() && !isCurrency() && !isKualiUser() && !isLookupHidden() && !isLookupReadonly() && !isWorkflowWorkgroup() && !isFile()&& !isButton() && !isLink()) {
            throw new CompletionException("error validating " + rootBusinessObjectClass.getName() + " control: unknown control type in control definition (" + "" + ")");
        }
        if (valuesFinderClass != null) {
        	try {
        		final Class<?> valuesFinderClassObject = ClassUtils.getClass(ClassLoaderUtils.getDefaultClassLoader(), getValuesFinderClass());
        		if (!KeyValuesFinder.class.isAssignableFrom(valuesFinderClassObject)) {
        			throw new ClassValidationException("valuesFinderClass is not a valid instance of " + KeyValuesFinder.class.getName() + " instead was: " + valuesFinderClassObject.getName());
        		}

                if (!CollectionUtils.isEmpty(valuesFinderProperties)) {
                    final Object finder = valuesFinderClassObject.newInstance();
                    final Set<String> invalid = valuesFinderProperties.keySet()
                            .stream()
                            .filter(k -> !PropertyUtils.isWriteable(finder, k))
                            .collect(Collectors.toSet());

                    if (!invalid.isEmpty()) {
                        throw new ClassValidationException(valuesFinderClassObject.getName() + " does not have the following writable properties: " + invalid);
                    }
                }
        	} catch (ClassNotFoundException|IllegalAccessException|InstantiationException e) {
        		throw new ClassValidationException("valuesFinderClass could not be found: " + getValuesFinderClass(), e);
        	}
        }
        if (getBusinessObjectClass() != null) {
        	try {
        		Class<?> businessObjectClassObject = ClassUtils.getClass(ClassLoaderUtils.getDefaultClassLoader(), getBusinessObjectClass());
        		if (!BusinessObject.class.isAssignableFrom(businessObjectClassObject)) {
        			throw new ClassValidationException("dataObjectClass is not a valid instance of " + BusinessObject.class.getName() + " instead was: " + businessObjectClassObject.getName());
        		}
        	} catch (ClassNotFoundException e) {
        		throw new ClassValidationException("dataObjectClass could not be found: " + getBusinessObjectClass(), e);
        	}
        }
    }


    @Override
    public String getScript() {
        return script;
    }

    /**
     * JavaScript script to run when a select control's value is changed.
     * 
     */
    @Override
    public void setScript(String script) {
        this.script = script;
    }

    @Override
    public boolean isRanged() {
		return this.ranged;
	}

    /**
     * Sets the control as a ranged (from and to) date field if true, or a single date field if false
     * 
     * @param ranged boolean true for a ranged control, false for a single date field
     */
	public void setRanged(boolean ranged) {
		this.ranged = ranged;
	}

    @Override
    public boolean equals(Object object) {
    	if ( !(object instanceof ControlDefinitionBase) ) {
    		return false;
    	}
    	ControlDefinitionBase rhs = (ControlDefinitionBase)object;
    	return new EqualsBuilder()
    	        .append( this.cols, rhs.cols )
    			.append( this.valuesFinderClass, rhs.valuesFinderClass )
    			.append( this.valuesFinderProperties, rhs.valuesFinderProperties )
    			.append( this.rows, rhs.rows )
    			.append( this.script, rhs.script )
    			.append( this.size, rhs.size )
    			.append( this.datePicker, rhs.datePicker )
    			.append( this.ranged, rhs.ranged )
    			.isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder()
                .append( this.cols)
                .append( this.valuesFinderClass )
                .append( this.valuesFinderProperties )
                .append( this.rows )
                .append( this.script )
                .append( this.size )
                .append( this.datePicker )
                .append( this.ranged )
                .toHashCode();
    }
}
