/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.util;

import org.apache.commons.codec.EncoderException;
import org.apache.commons.codec.net.URLCodec;
import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import java.nio.charset.StandardCharsets;
import java.util.Properties;
import java.util.regex.Pattern;

/**
 * Provides utility methods for re/building URLs.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class UrlFactory {
    private static Logger LOG = LogManager.getLogger(UrlFactory.class);
    private static Pattern NEEDS_ENCODING_PATTERN = Pattern.compile(".*[^a-zA-Z0-9-_.* |%\\d\\d].*");

    /**
     * Creates a new URL by taking the given URL and appending the parameter names and values from the given Properties instance to
     * it. Note: parameter names must be non-blank; parameter values must be non-null.
     * 
     * @param baseUrl the URL string used as the basis for reconstruction
     * @param params Properties instance containing the desired parameters and their values
     * @throws IllegalArgumentException if the given url is null or empty
     * @throws IllegalArgumentException if the given Properties instance is null
     * @throws IllegalArgumentException if a parameter name is null or empty, or a parameter value is null
     * @throws RuntimeException if there is a problem encoding a parameter name or value into UTF-8
     * @return a newly-constructed URL string which has the given parameters and their values appended to it
     */
    private static URLCodec urlCodec = new URLCodec(StandardCharsets.UTF_8.name());
    
    public static String parameterizeUrl(String baseUrl, Properties params) {
        baseUrl = StringUtils.trim(baseUrl);
        if (StringUtils.isEmpty(baseUrl)) {
            throw new IllegalArgumentException("invalid (blank) base URL");
        }
        if (params == null) {
            throw new IllegalArgumentException("invalid (null) Properties");
        }


        StringBuffer ret = new StringBuffer(baseUrl);
        // Only start with ? if it has not been added to the url
        String delimiter = (ret.indexOf("?") == -1) ? "?" : "&";
        for ( Object key : params.keySet() ) {
            String paramName = StringUtils.trim( (String)key );
            String paramValue = params.getProperty(paramName);
            ret.append( delimiter );
            if (StringUtils.isEmpty(paramName)) {
                throw new IllegalArgumentException("invalid (blank) paramName");
            }

            ret.append(safeEncode(paramName));
            ret.append("=");
            ret.append(safeEncode(paramValue));

            delimiter = "&";
        }

        return ret.toString();
    }

    @Deprecated
    public static String encode( String value ) {
        try {
            return urlCodec.encode(value);
        } catch ( EncoderException ex ) {
            throw new RuntimeException( "Unable to encode value: " + value, ex );
        }
    }

    //Attempt to encode this value only if it doesn't already seem to be encoded as determined by a decode call
    public static String safeEncode( String value ) {
        if (StringUtils.isEmpty(value)) {
            return "";
        }

        try {
            boolean needsEncoding = NEEDS_ENCODING_PATTERN.matcher(value).matches();
            return needsEncoding ? urlCodec.encode(value) : value;
        } catch (EncoderException e) {
            LOG.debug("Unable to encode value: " + value, e);
            throw new RuntimeException( "Unable to encode value: " + value, e );
        }
    }
}
