/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.data.metadata.impl;

import java.util.Collections;
import java.util.List;

import org.kuali.rice.krad.data.metadata.DataObjectCollection;
import org.kuali.rice.krad.data.metadata.DataObjectCollectionSortAttribute;

/**
* Collection meta data.
*
* <p>
* Implementation that represents the meta data for a collection in a data object.
* </p>
*
* @author Kuali Rice Team (rice.collab@kuali.org)
*/
public class DataObjectCollectionImpl extends MetadataChildBase implements DataObjectCollection {
	private static final long serialVersionUID = -785119931770775640L;

	protected DataObjectCollection embeddedCollection;

    protected String elementLabel;
	protected Long minItems;
	protected Long maxItems;
	protected List<DataObjectCollectionSortAttribute> defaultOrdering;
	protected Boolean indirectCollection;

    /**
    * The elementLabel defines the name to be used for a single object within the collection.
    *
    * <p>
    * For example: "Address" may be the name of one object within the "Addresses" collection.
    * </p>
    */
    @Override
	public String getElementLabel() {
		if (elementLabel == null) {
			elementLabel = getLabelFromPropertyName(relatedType.getSimpleName());
		}
        return elementLabel;
    }

    /**
    * Sets name used for single object within collection.
    *
    * @param elementLabel single object name
    */
    public void setElementLabel(String elementLabel) {
        this.elementLabel = elementLabel;
    }

	@Override
	public Long getMinItems() {
		if (minItems != null) {
			return minItems;
		}
		if (embeddedCollection != null) {
			return embeddedCollection.getMinItems();
		}
		return null;
    }

    /**
    * Sets minimum items in collection.
    *
    * @param minOccurs minimum items in collection.
    */
	public void setMinItemsInCollection(Long minOccurs) {
		this.minItems = minOccurs;
    }

    /**
    * {@inheritDoc}
    */
	@Override
	public Long getMaxItems() {
		if (maxItems != null) {
			return maxItems;
		}
		if (embeddedCollection != null) {
			return embeddedCollection.getMaxItems();
		}
		return null;
    }

    /**
    * Sets maximum items in collection.
    *
    * @param maxOccurs maximum items in collection.
    */
	public void setMaxItemsInCollection(Long maxOccurs) {
		this.maxItems = maxOccurs;
    }

    /**
    * {@inheritDoc}
    */
	@Override
	public List<DataObjectCollectionSortAttribute> getDefaultOrdering() {
		if (defaultOrdering != null) {
			return defaultOrdering;
		}
		if (embeddedCollection != null) {
			return embeddedCollection.getDefaultOrdering();
		}
		return Collections.emptyList();
	}

    /**
    * Sets attribute that the default order of the collection.
    *
    * @param defaultCollectionOrdering attribute name
    */
	public void setDefaultCollectionOrderingAttributeNames(
			List<DataObjectCollectionSortAttribute> defaultCollectionOrdering) {
		this.defaultOrdering = defaultCollectionOrdering;
	}

    /**
    * {@inheritDoc}
    */
	@Override
	public boolean isIndirectCollection() {
		if (indirectCollection != null) {
			return indirectCollection;
		}
		if (embeddedCollection != null) {
			return embeddedCollection.isIndirectCollection();
		}
		return false;
	}

    /**
    * Sets whether linked item is used.
    *
    * @param indirectCollection whether link item used.
    */
	public void setIndirectCollection(boolean indirectCollection) {
		this.indirectCollection = indirectCollection;
	}

    /**
    * Gets the embedded collection.
    *
    * @return the embedded collection, if it exists.
    */
	public DataObjectCollection getEmbeddedCollection() {
		return embeddedCollection;
	}

	public void setEmbeddedCollection(DataObjectCollection embeddedCollection) {
		this.embeddedCollection = embeddedCollection;
		setEmbeddedMetadataChild(embeddedCollection);
	}

}
