/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.rice.krad.data.jpa;

import org.apache.commons.lang.StringUtils;
import org.eclipse.persistence.internal.jpa.querydef.PathImpl;

import javax.persistence.EntityManager;
import javax.persistence.criteria.*;
import javax.persistence.criteria.CriteriaQuery;
import java.util.ArrayList;
import java.util.List;

/**
 * Thin abstraction/container for criteria parsing context.
 */
class TranslationContext {

    /**
     * The builder for the criteria.
     */
    CriteriaBuilder builder;

    /**
     * The query for the criteria.
     */
    CriteriaQuery<?> query;

    /**
     * The FROM clause root type.
     */
    Root root;

    /**
     * The list of predicates.
     */
    List<Predicate> predicates = new ArrayList<>();

    /**
     * If the current context is a sub-query of another context, the parent context will be stored here.
     */
    TranslationContext parentTranslationContext;

    public static TranslationContext createCriteria(EntityManager entityManager, Class<?> queryClass) {
        return new TranslationContext(entityManager, queryClass, false);
    }

    public static TranslationContext createCriteriaForSubQuery(EntityManager entityManager, Class<?> queryClass, TranslationContext parentContext) {
        return new TranslationContext(entityManager, queryClass, parentContext);
    }

    public static TranslationContext createInnerCriteria(TranslationContext parent) {
        return new TranslationContext(parent);
    }

    public static TranslationContext createCriteriaForCount(EntityManager entityManager, Class<?> queryClass) {
        return new TranslationContext(entityManager, queryClass, true);
    }

    /**
     * Creates a new criteria parsing context.
     *
     * @param entityManager the entity manager to use for interacting with the database.
     * @param queryClass    the type of the query.
     */
    private TranslationContext(EntityManager entityManager, Class<?> queryClass, boolean count) {
        builder = entityManager.getCriteriaBuilder();
        if (count) {
            CriteriaQuery<Long> q = builder.createQuery(Long.class);
            Root<?> r = q.from(queryClass);
            q.select(builder.count(r));
            query = q;
            root = r;
        } else {
            query = builder.createQuery(queryClass);
            // establish single consistent root instance
            // we need this or erroneous query against distinct table aliases is generated
            root = query.from(query.getResultType());
        }
    }

    /**
     * Creates a new criteria parsing context for an inner subquery.  The parent context is stored
     * to allow references between the inner and outer queries.
     *
     * @param entityManager the entity manager to use for interacting with the database.
     * @param queryClass    the type of the query.
     * @param parentContext The {@link TranslationContext} of the outer query into which the subquery will be added as a {@link Predicate}.
     */
    private TranslationContext(EntityManager entityManager, Class<?> queryClass, TranslationContext parentContext) {
        this(entityManager, queryClass, false);
        this.parentTranslationContext = parentContext;
    }

    /**
     * Creates a new criteria parsing context that is a container for the inner predicates.
     *
     * @param parent the parent criteria parsing context.
     */
    private TranslationContext(TranslationContext parent) {
        builder = parent.builder;
        query = parent.query;
        root = parent.root;
        parentTranslationContext = parent.parentTranslationContext;
    }

    /**
     * Adds a predicate.
     *
     * @param predicate the predicate to add.
     */
    void addPredicate(Predicate predicate) {
        predicates.add(predicate);
    }

    /**
     * Adds an AND clause.
     *
     * @param predicate the predicate to AND.
     */
    void and(TranslationContext predicate) {
        addPredicate(predicate.getCriteriaPredicate());
    }

    /**
     * Adds a JPA Subquery to the predicates.
     *
     * @param subquery the subquery to add.
     */
    void addExistsSubquery(Subquery<?> subquery) {
        predicates.add(builder.exists(subquery));
    }

    /**
     * Adds an OR clause.
     *
     * @param predicate the predicate to OR.
     */
    void or(TranslationContext predicate) {
        List<Predicate> newpredicates = new ArrayList<>();
        //When traversed to a simple OR predicate you may not have a criteria predicate set so check and just
        //add to builder if necKradEclipseLinkEntityManagerFactoryessary
        Predicate criteriaPredicate = getCriteriaPredicate();
        Predicate orPredicate;
        if (criteriaPredicate != null) {
            orPredicate = builder.or(new Predicate[]{predicate.getCriteriaPredicate(), getCriteriaPredicate()});
        } else {
            orPredicate = builder.or(predicate.getCriteriaPredicate());
        }
        newpredicates.add(orPredicate);
        predicates = newpredicates;
    }

    /**
     * Gets the criteria predicate.
     *
     * @return the criteria predicate.
     */
    Predicate getCriteriaPredicate() {
        if (predicates.size() == 1) {
            return predicates.get(0);
        } else if (predicates.size() > 1) {
            return builder.and(predicates.toArray(new Predicate[0]));
        } else {
            return null;
        }
    }

    /**
     * Gets the path for the given attribute.
     *
     * @param attr the attribute path.
     * @return the path for the given attribute.
     */
    <T> Expression<T> attr(String attr) {
        if (StringUtils.isBlank(attr)) {
            throw new IllegalArgumentException("Encountered an empty attribute path");
        }

        Path<T> path = root;

        if (!attr.contains(".")) {
            //  There is no nested property path so no need to worry about any joins
            path = path.get(attr);
        } else {
            // Tokenize the property string
            String[] attrArray = attr.split("\\.");
            // first, check if this is a reference to a field on the parent (outer) query.
            // If so, and we have a parent (outer) query, then strip off the parent keyword
            // and resolve the property in that context.
            if (attrArray.length > 0 && StringUtils.equals(attrArray[0], "parent") && parentTranslationContext != null) {
                return parentTranslationContext.attr(StringUtils.substringAfter(attr, "."));
            } else {
                // split the attribute based on a period for nested property paths, for example if you want to
                // pass an attribute like "property1.property2" then JPA will not interpret that properly,
                // you have to split it manually
                for (String attrElement : attrArray) {
                    if (StringUtils.isBlank(attrElement)) {
                        throw new IllegalArgumentException("Encountered an empty path element in property path: "
                                + attr);
                    }

                    boolean existingJoinFoundForThisAttrElement = false;

                    // If there is already a join for this attribute then use it as the path
                    List<PathImpl<T>> listOfJoins = new ArrayList<>(root.getJoins());

                    for (PathImpl<T> existingJoin : listOfJoins) {
                        if (existingJoin.getCurrentNode().getName().equalsIgnoreCase(attrElement)) {
                            path = existingJoin;
                            existingJoinFoundForThisAttrElement = true;
                        }
                    }

                    if (!existingJoinFoundForThisAttrElement) {
                        path = path.get(attrElement);
                    }
                }
            }
        }

        return path;
    }
}
