/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.data.jpa.converters;

import javax.persistence.AttributeConverter;
import javax.persistence.Converter;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.CoreApiServiceLocator;
import org.kuali.rice.core.api.encryption.EncryptionService;

/**
 * Calls the core service to hash values going to the database.
 * 
 *  @author Kuali Rice Team (rice.collab@kuali.org)
 */
@Converter
public class HashConverter implements AttributeConverter<String, String> {

    /**
     * {@inheritDoc}
     *
     * This implementation hashes the value going to the database.
     */
	@Override
	public String convertToDatabaseColumn(String objectValue) {
		// don't attempt to encrypt nulls or empty strings
		if (objectValue == null) {
			return null;
		}
		if (StringUtils.isEmpty(objectValue.toString())) {
			return "";
		}
		// don't convert if already a hashed value
		if (objectValue.toString().endsWith(EncryptionService.HASH_POST_PREFIX)) {
			return StringUtils.stripEnd(objectValue.toString(), EncryptionService.HASH_POST_PREFIX);
		} else {
			try {
				return CoreApiServiceLocator.getEncryptionService().hash(objectValue);
			} catch (Exception e) {
				throw new RuntimeException("Exception while attempting to hash value for DB: ", e);
			}
		}
	}

    /**
     * {@inheritDoc}
     *
     * This implementation directly returns the hash value coming from the database.
     */
	@Override
	public String convertToEntityAttribute(String dataValue) {
		// don't attempt to decrypt nulls or empty strings
		if (dataValue == null) {
			return null;
		}
		if (StringUtils.isEmpty(dataValue.toString())) {
			return "";
		}
		return dataValue.toString() + EncryptionService.HASH_POST_PREFIX;
	}

}
