/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.data;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * Stores the values for a multi-valued key. This is intended primarily for use on
 * {@link DataObjectService#find(Class, Object)} in situations where you have a data object which has a compound
 * primary key represented by more than one field. In such cases the keys in the map you construction this class with
 * should be the field names of the primary key fields, and the values in the maps should be the values by which you
 * want to perform the find.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public final class CompoundKey implements Serializable {

	private static final long serialVersionUID = 1L;

    private final Map<String, ?> keys;

    /**
     * Construct a new instance of a CompoundKey from the given key values map.
     *
     * @param keys map of field name to value for the compound key, must be non-null and non-empty
     *
     * @throws IllegalArgumentException if the given Map is null or empty
     */
    public CompoundKey(Map<String, ?> keys) {
        if (keys == null || keys.isEmpty()) {
            throw new IllegalArgumentException("Compound key map should be non-null as well as having at least one"
                    + "value.");
        }
        this.keys = new HashMap<String, Object>(keys);
    }

    /**
     * Returns an unmodifable Map of the key values on this CompoundKey
     *
     * @return unmodifiable map of the key values on this CompoundKey
     */
    public Map<String, ?> getKeys() {
        return Collections.unmodifiableMap(keys);
    }

	/**
	 * Returns true if any of the fields in this compound key have null values, since that usually indicates an
	 * incomplete and unsaved object.
	 * 
	 * @return
	 */
	public boolean hasNullKeyValues() {
		for (Object value : keys.values()) {
			if (value == null) {
				return true;
			}
		}
		return false;
	}

}
