/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.service.impl;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.reflection.ObjectAccessException;
import com.thoughtworks.xstream.converters.reflection.PureJavaReflectionProvider;
import com.thoughtworks.xstream.converters.reflection.ReflectionConverter;
import com.thoughtworks.xstream.converters.reflection.ReflectionProvider;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.mapper.Mapper;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.rice.core.framework.security.XStreamSecurityService;
import org.kuali.rice.krad.service.LegacyDataAdapter;
import org.kuali.rice.krad.service.XmlObjectSerializerService;
import org.kuali.rice.krad.service.util.DateTimeConverter;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;

import java.lang.reflect.Field;
import java.util.*;


/**
 * Service implementation for the XmlObjectSerializer structure. This is the default implementation that gets
 * delivered with Kuali. It utilizes the XStream open source libraries and framework.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class XmlObjectSerializerServiceImpl implements XmlObjectSerializerService, InitializingBean {
    private static final Logger LOG = LogManager.getLogger(XmlObjectSerializerServiceImpl.class);

    protected LegacyDataAdapter lda;
    protected XStreamSecurityService xStreamSecurityService;
	protected XStream xstream;

    @Required
    public void setLegacyDataAdapter(LegacyDataAdapter lda) {
        this.lda = lda;
    }

    @Required
    public void setXStreamSecurityService(XStreamSecurityService xStreamSecurityService) {
        this.xStreamSecurityService = xStreamSecurityService;
    }

    @Override
	public String toXml(Object object) {
    	if ( LOG.isDebugEnabled() ) {
    		LOG.debug( "toXml(" + object + ") : \n" + xstream.toXML(object) );
    	}
        return xstream.toXML(object);
    }

    @Override
	public Object fromXml(String xml) {
    	if ( LOG.isDebugEnabled() ) {
    		LOG.debug( "fromXml() : \n" + xml );
    	}
    	if ( xml != null ) {
    		xml = xml.replaceAll( "\\.ByteBuddyOJBProxy", "" );
    	}
        return xstream.fromXML(xml);
    }

    @Override
    public void afterPropertiesSet() {
        xstream = xStreamSecurityService.applySecurityRules(new XStream(new ProxyAwareJavaReflectionProvider()));
        xstream.registerConverter(new ProxyConverter(xstream.getMapper(), xstream.getReflectionProvider() ));
        try {
            Class<?> objListProxyClass = Class.forName("org.apache.ojb.broker.core.proxy.ListProxyDefaultImpl");
            xstream.addDefaultImplementation(ArrayList.class, objListProxyClass);
        } catch ( Exception ex ) {
            // Do nothing - this will blow if the OJB class does not exist, which it won't in some installs
        }
        xstream.registerConverter(new DateTimeConverter());
    }

    /**
     * This custom converter only handles proxies for BusinessObjects.  List-type proxies are handled by configuring XStream to treat
     * ListProxyDefaultImpl as ArrayLists (see constructor for this service).
     */
    public class ProxyConverter extends ReflectionConverter {
        public ProxyConverter(Mapper mapper, ReflectionProvider reflectionProvider) {
            super(mapper, reflectionProvider);
        }

        @Override
        public boolean canConvert(Class clazz) {
            return clazz.getName().contains("ByteBuddyOJBProxy") || clazz.getName().equals("org.apache.ojb.broker.core.proxy.ListProxyDefaultImpl");
        }

        @Override
        public void marshal(Object obj, HierarchicalStreamWriter writer, MarshallingContext context) {
            if (obj.getClass().getName().equals("org.apache.ojb.broker.core.proxy.ListProxyDefaultImpl")) {
                context.convertAnother( new ArrayList<>((List<?>) obj) );
            }
            else {
                super.marshal(lda.resolveProxy(obj), writer, context);
            }
        }

        // we shouldn't need an unmarshal method because all proxy metadata is taken out of the XML, so we'll reserialize as a base BO.
    }

    public class ProxyAwareJavaReflectionProvider extends PureJavaReflectionProvider {

        /**
         * @see com.thoughtworks.xstream.converters.reflection.PureJavaReflectionProvider#visitSerializableFields(java.lang.Object, com.thoughtworks.xstream.converters.reflection.ReflectionProvider.Visitor)
         */
        @Override
        public void visitSerializableFields(Object object, Visitor visitor) {
            for (Iterator<?> iterator = fieldDictionary.serializableFieldsFor(object.getClass()); iterator.hasNext();) {
                Field field = (Field) iterator.next();
                if (!fieldModifiersSupported(field)) {
                    continue;
                }
                validateFieldAccess(field);
                if (ignoreField(field)) {
                    continue;
                }
                Object value;
                try {
                    value = field.get(object);
                    if (value != null && lda.isProxied(value)) {
                        value = lda.resolveProxy(value);
                    }
                } catch (Exception e) {
                    throw new ObjectAccessException("Could not get field " + field.getDeclaringClass() + "." + field.getName() + " on " + object, e);
                }
                visitor.visit(field.getName(), field.getType(), field.getDeclaringClass(), value);
            }
        }

        protected boolean ignoreField(Field field) {
            return false;
        }

    }

}
