/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.service.impl;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.criteria.PredicateFactory;
import org.kuali.rice.core.api.criteria.QueryByCriteria;
import org.kuali.rice.core.api.criteria.QueryResults;
import org.kuali.rice.kim.api.group.Group;
import org.kuali.rice.kim.api.services.KimApiServiceLocator;
import org.kuali.rice.krad.bo.AdHocRoutePerson;
import org.kuali.rice.krad.bo.AdHocRouteRecipient;
import org.kuali.rice.krad.bo.AdHocRouteWorkgroup;
import org.kuali.rice.krad.data.DataObjectService;
import org.kuali.rice.krad.document.Document;
import org.kuali.rice.krad.service.DocumentAdHocService;

/**
 * Implementation for {@link DocumentAdHocService}
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 *
 */
public class DocumentAdHocServiceImpl implements DocumentAdHocService {

	protected DataObjectService dataObjectService;

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void replaceAdHocsForDocument(Document document) {
		if ( document == null || StringUtils.isBlank(document.getDocumentNumber())) {
			return;
		}
		dataObjectService.deleteMatching(AdHocRoutePerson.class, QueryByCriteria.Builder.forAttribute("documentNumber",
				document.getDocumentNumber()).build());
		dataObjectService.deleteMatching(AdHocRouteWorkgroup.class, QueryByCriteria.Builder.forAttribute(
				"documentNumber", document.getDocumentNumber()).build());



		document.setAdHocRoutePersons(saveAdHocRouteRecipients(document.getDocumentNumber(),
				document.getAdHocRoutePersons()));
		document.getAdHocRoutePersons().forEach(AdHocRoutePerson::getPerson);

		document.setAdHocRouteWorkgroups(saveAdHocRouteRecipients(document.getDocumentNumber(),
				document.getAdHocRouteWorkgroups()));
		document.getAdHocRouteWorkgroups().forEach(adHocRouteWorkgroup -> adHocRouteWorkgroup.setId(adHocRouteWorkgroup.getId()));
	}

	protected <T extends AdHocRouteRecipient> List<T> saveAdHocRouteRecipients(String documentNumber, List<T> adHocRouteRecipients) {
		final List<T> savedAdHocRouteWorkgroup = new ArrayList<>();
		if (adHocRouteRecipients != null) {
			for (T recipient : adHocRouteRecipients) {
				recipient.setdocumentNumber(documentNumber);
				savedAdHocRouteWorkgroup.add(dataObjectService.save(recipient));
			}
		}
		return savedAdHocRouteWorkgroup;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addAdHocs(Document document) {
		if ( document == null ) {
			return;
		}
        /* Instead of reading the doc header to see if doc is in saved status
         * its probably easier and faster to just do this all the time and
         * store a null when appropriate.
         */
        QueryResults<AdHocRoutePerson> adHocRoutePersons = dataObjectService.findMatching(AdHocRoutePerson.class,
			    QueryByCriteria.Builder.fromPredicates(
			    		PredicateFactory.equal("documentNumber", document.getDocumentNumber()),
			    		PredicateFactory.equal("type", AdHocRoutePerson.PERSON_TYPE) ) );

        QueryResults<AdHocRouteWorkgroup> adHocRouteWorkgroups = dataObjectService.findMatching(AdHocRouteWorkgroup.class,
			    QueryByCriteria.Builder.fromPredicates(
			    		PredicateFactory.equal("documentNumber", document.getDocumentNumber()),
			    		PredicateFactory.equal("type", AdHocRoutePerson.WORKGROUP_TYPE) ) );

        //populate group namespace and names on adHocRouteWorkgroups
        for (AdHocRouteWorkgroup adHocRouteWorkgroup : adHocRouteWorkgroups.getResults()) {
            Group group = KimApiServiceLocator.getGroupService().getGroup(adHocRouteWorkgroup.getId());
            adHocRouteWorkgroup.setRecipientName(group.getName());
            adHocRouteWorkgroup.setRecipientNamespaceCode(group.getNamespaceCode());
        }

        // We *must* copy these into new arrays.  The returned lists are unmodifiable.
        document.setAdHocRoutePersons( new ArrayList<AdHocRoutePerson>( adHocRoutePersons.getResults() ) );
        document.setAdHocRouteWorkgroups( new ArrayList<AdHocRouteWorkgroup>( adHocRouteWorkgroups.getResults() ) );
	}

	public void setDataObjectService(DataObjectService dataObjectService) {
		this.dataObjectService = dataObjectService;
	}

}
