/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.datadictionary.mask;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.krad.datadictionary.parse.BeanTag;
import org.kuali.rice.krad.datadictionary.parse.BeanTagAttribute;

/**
 * The maskTo element is to used hide the beginning part of the value for
 * unauthorized users. The number of leading characters to hide and the
 * replacement character can be specified.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@BeanTag(name = "maskFormatteSubString")
public class MaskFormatterSubString implements MaskFormatter {
    private static final long serialVersionUID = -876112522775686636L;

    protected String maskCharacter = "*";
    protected int maskLength;

    @Override
    public String maskValue(Object value) {
        if (value == null) {
            return null;
        }

        // TODO: MOVE TO UNIT TEST: move this validation into the unit tests
        if (maskCharacter == null) {
            throw new RuntimeException("Mask character not specified. Check DD maskTo attribute.");
        }

        String strValue = value.toString();
        if (strValue.length() < maskLength) {
            return StringUtils.repeat(maskCharacter, maskLength);
        }
        if (maskLength > 0) {
            return StringUtils.repeat(maskCharacter, maskLength) + strValue.substring(maskLength);
        } else {
            return strValue;
        }
    }

    /**
     * Gets the maskCharacter attribute.
     *
     * @return Returns the maskCharacter.
     */
    @BeanTagAttribute(name = "maskCharacter")
    public String getMaskCharacter() {
        return maskCharacter;
    }

    /**
     * Specify the character with which to mask the original value.
     *
     * @param maskCharacter for masking values
     */
    public void setMaskCharacter(String maskCharacter) {
        this.maskCharacter = maskCharacter;
    }

    /**
     * Gets the maskLength attribute.
     *
     * @return Returns the maskLength.
     */
    @BeanTagAttribute(name = "maskLength")
    public int getMaskLength() {
        return maskLength;
    }

    /**
     * Set the number of characters to mask at the beginning of the string.
     *
     * @param maskLength The maskLength to set.
     */
    public void setMaskLength(int maskLength) {
        this.maskLength = maskLength;
    }

}
