/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.datadictionary.validation.constraint.provider;

import org.kuali.rice.krad.datadictionary.validation.capability.Constrainable;
import org.kuali.rice.krad.datadictionary.validation.constraint.Constraint;
import org.kuali.rice.krad.datadictionary.validation.constraint.resolver.ConstraintResolver;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * BaseConstraintProvider implements a simple in memory storage map of constraint resolvers
 *
 * <p>This provides a convenient base class
 * from which other constraint providers can be derived.</p>
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 * @param <T> constrainable type
 * @since 1.1
 */
public abstract class BaseConstraintProvider<T extends Constrainable> implements ConstraintProvider<T> {

    protected Map<String, ConstraintResolver<T>> resolverMap;

    /**
     * initializes the constraints
     *
     * <p>By doing initialization here, and not in a constructor, constraints are only placed in memory when they are
     * utilized.</p>
     */
    public void init() {
        if (resolverMap == null) {
            resolverMap = new HashMap<String, ConstraintResolver<T>>();
        }

    }

    /**
     * @see org.kuali.rice.krad.datadictionary.validation.constraint.provider.ConstraintProvider#getConstraints(org.kuali.rice.krad.datadictionary.validation.capability.Constrainable,
     *      java.lang.Class)
     */
    @Override
    public List<Constraint> getConstraints(T definition, Class<? extends Constraint> constraintType) {
        if (resolverMap == null) {
            init();
        }

        ConstraintResolver<T> resolver = resolverMap.get(constraintType.getName());

        if (resolver == null) {
            return null;
        }

        return resolver.resolve(definition);
    }

    /**
     * @return the resolverMap
     */
    public Map<String, ConstraintResolver<T>> getResolverMap() {
        return this.resolverMap;
    }

    /**
     * @param resolverMap the resolverMap to set
     */
    public void setResolverMap(Map<String, ConstraintResolver<T>> resolverMap) {
        this.resolverMap = resolverMap;
    }

}
