/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.datadictionary.validation.constraint.provider;

import java.util.List;

import org.kuali.rice.krad.datadictionary.validation.capability.CaseConstrainable;
import org.kuali.rice.krad.datadictionary.validation.capability.Constrainable;
import org.kuali.rice.krad.datadictionary.validation.constraint.Constraint;

/**
 * ConstraintProvider determines a list of constraints for a given Constrainable definition for an attribute
 * in the data dictionary
 *
 * <p>The ConstraintProvider interface must be implemented by any class that contributes
 * Constraints to the DictionaryValidationService. Multiple ConstraintProviders can be registered simultaneously,
 * and each can contribute constraints for any number of constraint types.</p>
 *
 * <p>
 * These constraints can be looked up in a variety of ways. They may be:</p>
 * <ol>
 * <li> member variables of the Constrainable definition itself {@link CaseConstrainable}</li>
 * <li> provided from some external source, or generated on the fly</li>
 * </ol>
 *
 * <p>The goal here is to provide a mechanism that enables implementing institutions to inject new Constraints and
 * ConstraintProcessor
 * classes into the DictionaryValidationService implementation via dependency injection.</p>
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 * @param <T> constrainable type
 * @since 1.1
 */
public interface ConstraintProvider<T extends Constrainable> {

    /**
     * gets the constraints provided
     *
     * @param definition - a Data Dictionary definition e.g. {@code ComplexAttributeDefinition} or {@code
     * CollectionDefinition}
     * @param constraintType - a java class that represents the constraint
     * @return the list of constraints
     */
    public List<Constraint> getConstraints(T definition, Class<? extends Constraint> constraintType);

    /**
     * checks whether this provider supports the provided definition
     *
     * @param definition - a Data Dictionary definition e.g. {@code AttributeDefinition}
     * @return true if supported, false otherwise
     */
    public boolean isSupported(Constrainable definition);

}
