/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.datadictionary.validation.processor;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.krad.datadictionary.exception.AttributeValidationException;
import org.kuali.rice.krad.datadictionary.validation.AttributeValueReader;
import org.kuali.rice.krad.datadictionary.validation.ValidationUtils;
import org.kuali.rice.krad.datadictionary.validation.constraint.Constraint;
import org.kuali.rice.krad.datadictionary.validation.constraint.PrerequisiteConstraint;
import org.kuali.rice.krad.datadictionary.validation.result.ConstraintValidationResult;
import org.kuali.rice.krad.datadictionary.validation.result.DictionaryValidationResult;
import org.kuali.rice.krad.datadictionary.validation.result.ProcessorResult;
import org.kuali.rice.krad.uif.UifConstants;

/**
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class PrerequisiteConstraintProcessor extends BasePrerequisiteConstraintProcessor<PrerequisiteConstraint> {

    private static final String CONSTRAINT_NAME = "prerequisite constraint";
    private static final String FALLBACK_KEY = "prerequisiteFallback";

    /**
     * {@inheritDoc}
     */
    @Override
    public ProcessorResult process(DictionaryValidationResult result, Object value, PrerequisiteConstraint constraint,
            AttributeValueReader attributeValueReader) throws AttributeValidationException {

        if (ValidationUtils.isNullOrEmpty(value)) {
            return new ProcessorResult(result.addSkipped(attributeValueReader, CONSTRAINT_NAME));
        }

        ConstraintValidationResult constraintValidationResult = processPrerequisiteConstraint(constraint,
                attributeValueReader);

        if (constraint != null) {
            if (StringUtils.isNotBlank(constraint.getMessageKey())) {
                constraintValidationResult.setConstraintLabelKey(constraint.getMessageKey());
            } else {
                constraintValidationResult.setConstraintLabelKey(
                        UifConstants.Messages.VALIDATION_MSG_KEY_PREFIX + FALLBACK_KEY);
            }
            constraintValidationResult.setErrorParameters(constraint.getValidationMessageParamsArray());
        }

        result.addConstraintValidationResult(attributeValueReader, constraintValidationResult);

        return new ProcessorResult(constraintValidationResult);
    }

    @Override
    public String getName() {
        return CONSTRAINT_NAME;
    }

    /**
     * @see org.kuali.rice.krad.datadictionary.validation.processor.ConstraintProcessor#getConstraintType()
     */
    @Override
    public Class<? extends Constraint> getConstraintType() {
        return PrerequisiteConstraint.class;
    }

}
