/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.datadictionary.validation.processor;

import org.kuali.rice.krad.datadictionary.exception.AttributeValidationException;
import org.kuali.rice.krad.datadictionary.validation.AttributeValueReader;
import org.kuali.rice.krad.datadictionary.validation.constraint.Constraint;
import org.kuali.rice.krad.datadictionary.validation.constraint.SimpleConstraint;
import org.kuali.rice.krad.datadictionary.validation.result.ConstraintValidationResult;
import org.kuali.rice.krad.datadictionary.validation.result.DictionaryValidationResult;
import org.kuali.rice.krad.datadictionary.validation.result.ProcessorResult;

import java.util.ArrayList;
import java.util.List;

/**
 * Processor for simple constraint which takes out each constraining value it contains and calls the appropriate
 * processor
 */
public class SimpleConstraintProcessor extends MandatoryElementConstraintProcessor<SimpleConstraint> {

    private static final String CONSTRAINT_NAME = "simple constraint";

    RangeConstraintProcessor rangeConstraintProcessor = new RangeConstraintProcessor();
    LengthConstraintProcessor lengthConstraintProcessor = new LengthConstraintProcessor();
    ExistenceConstraintProcessor existenceConstraintProcessor = new ExistenceConstraintProcessor();
    DataTypeConstraintProcessor dataTypeConstraintProcessor = new DataTypeConstraintProcessor();

    /**
     * Processes the SimpleConstraint by calling process on the other smaller constraints it represents and
     * putting the results together in ProcessorResult
     *
     * @return processor result
     * @throws AttributeValidationException
     * @see MandatoryElementConstraintProcessor#process(org.kuali.rice.krad.datadictionary.validation.result.DictionaryValidationResult,
     *      Object, org.kuali.rice.krad.datadictionary.validation.constraint.Constraint,
     *      org.kuali.rice.krad.datadictionary.validation.AttributeValueReader)
     */
    @Override
    public ProcessorResult process(DictionaryValidationResult result, Object value, final SimpleConstraint constraint,
            AttributeValueReader attributeValueReader) throws AttributeValidationException {

        ProcessorResult dataTypePR = dataTypeConstraintProcessor.process(result, value, constraint,
                attributeValueReader);
        ProcessorResult existencePR = existenceConstraintProcessor.process(result, value, constraint,
                attributeValueReader);
        ProcessorResult rangePR = rangeConstraintProcessor.process(result, value, constraint, attributeValueReader);
        ProcessorResult lengthPR = lengthConstraintProcessor.process(result, value, constraint, attributeValueReader);
        List<ConstraintValidationResult> cvrList = new ArrayList<ConstraintValidationResult>();
        cvrList.addAll(existencePR.getConstraintValidationResults());
        cvrList.addAll(rangePR.getConstraintValidationResults());
        cvrList.addAll(lengthPR.getConstraintValidationResults());
        cvrList.addAll(dataTypePR.getConstraintValidationResults());
        return new ProcessorResult(cvrList);
    }

    @Override
    public String getName() {
        return CONSTRAINT_NAME;
    }

    @Override
    public Class<? extends Constraint> getConstraintType() {
        return SimpleConstraint.class;
    }
}
