/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.inquiry;

import java.util.Set;

import org.kuali.rice.krad.bo.Exporter;
import org.kuali.rice.krad.datadictionary.DataObjectEntry;
import org.kuali.rice.krad.service.KRADServiceLocatorWeb;
import org.kuali.rice.krad.uif.view.InquiryView;
import org.kuali.rice.krad.uif.view.View;
import org.kuali.rice.krad.uif.view.ViewPresentationControllerBase;
import org.kuali.rice.krad.util.KRADConstants;
import org.kuali.rice.krad.web.form.InquiryForm;
import org.kuali.rice.krad.web.form.UifFormBase;

/**
 * Implementation of {@link org.kuali.rice.krad.uif.view.ViewPresentationController} for
 * {@link InquiryView} instances
 *
 * <p>
 * Adds flag for export of inquiry record
 * </p>
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class InquiryViewPresentationControllerBase extends ViewPresentationControllerBase {
    private static final long serialVersionUID = 7504225899471226403L;

    /**
     * @see org.kuali.rice.krad.uif.view.ViewPresentationController#getActionFlags(org.kuali.rice.krad.uif.view.View,
     * org.kuali.rice.krad.web.form.UifFormBase)
     */
    @Override
    public Set<String> getActionFlags(View view, UifFormBase model) {
        Set<String> actionFlags = super.getActionFlags(view, model);

        InquiryView inquiryView = (InquiryView) view;
        InquiryForm inquiryForm = (InquiryForm) model;

        if (isExportSupported(inquiryView, inquiryForm)) {
            actionFlags.add(KRADConstants.KUALI_ACTION_CAN_EXPORT);
        }

        return actionFlags;
    }

    /**
     * Examines the data objects data dictionary entry to determine if it supports XML export or not
     * @param view inquiry view
     *
     * @return boolean true if it supports export, false if not
     */
    protected boolean isExportSupported(InquiryView view, InquiryForm form) {
        Object dataObject = form.getDataObject();

        if (dataObject == null) {
            return false;
        }

        DataObjectEntry dataObjectEntry =
                KRADServiceLocatorWeb.getDataDictionaryService().getDataDictionary().getDataObjectEntry(
                        view.getDataObjectClassName().getName());

        Class<? extends Exporter> exporterClass = dataObjectEntry.getExporterClass();
        if (exporterClass != null) {
            try {
                Exporter exporter = exporterClass.newInstance();
                if (exporter.getSupportedFormats(dataObjectEntry.getDataObjectClass()).contains(
                        KRADConstants.XML_FORMAT)) {
                    return true;
                }
            } catch (Exception e) {
                throw new RuntimeException("Failed to locate or create exporter class: " + exporterClass, e);
            }
        }

        return false;
    }
}
