/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.uif.layout.collections;

import org.kuali.rice.krad.uif.UifConstants;
import org.kuali.rice.krad.uif.container.CollectionGroup;
import org.kuali.rice.krad.uif.util.ObjectPropertyUtils;
import org.kuali.rice.krad.uif.view.View;
import org.kuali.rice.krad.web.form.UifFormBase;

import java.util.List;

/**
 * StackedPagingHelper contains method(s) to help determine the correct page display information during
 * a request.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class CollectionPagingHelper {

    /**
     * Process the paging request by determining the displayStart value based on the page requested.
     *
     * @param view the current view
     * @param form the form
     * @param page the page requested (can be a number, prev, next, first, last)
     */
    public void processPagingRequest(View view, CollectionGroup collectionGroup, UifFormBase form, String page) {
        // allow only one concurrent request per view
        synchronized (view) {
            List<Object> modelCollection = ObjectPropertyUtils.getPropertyValue(form,
                    collectionGroup.getBindingInfo().getBindingPath());

            int displayStart = collectionGroup.getDisplayStart();
            int displayLength = collectionGroup.getDisplayLength();

            // adjust displayStart based on the page requested
            if (page.equals(UifConstants.PageRequest.FIRST)) {
                displayStart = 0;
            } else if (page.equals(UifConstants.PageRequest.PREV)) {
                displayStart = displayStart - displayLength;
            } else if (page.equals(UifConstants.PageRequest.NEXT)) {
                displayStart = displayStart + displayLength;
            } else if (page.equals(UifConstants.PageRequest.LAST)) {
                int lastPageSize = modelCollection.size() % displayLength;

                if (lastPageSize != 0) {
                    displayStart = modelCollection.size() - lastPageSize;
                } else {
                    displayStart = modelCollection.size() - displayLength;
                }
            } else {
                displayStart = ((Integer.parseInt(page.trim()) - 1) * displayLength);
            }

            collectionGroup.setDisplayStart(displayStart);
        }
    }
}
