/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.uif.lifecycle;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.krad.uif.UifConstants;
import org.kuali.rice.krad.uif.component.Component;
import org.kuali.rice.krad.uif.modifier.ComponentModifier;

/**
 * View lifecycle task to run component modifiers based on the lifecycle phase.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 * @see org.kuali.rice.krad.uif.modifier.ComponentModifier
 */
public class RunComponentModifiersTask extends ViewLifecycleTaskBase<Component> {

    public RunComponentModifiersTask() {
        super(Component.class);
    }

    /**
     * Runs any configured Component Modifiers for the given component that match the
     * given run phase and who run condition evaluation succeeds.
     *
     * {@inheritDoc}
     */
    @Override
    protected void performLifecycleTask() {
        Component component = (Component) getElementState().getElement();

        List<ComponentModifier> componentModifiers = component.getComponentModifiers();
        if (componentModifiers == null) {
            return;
        }

        Object model = ViewLifecycle.getModel();
        String runPhase = getElementState().getViewPhase();
        for (ComponentModifier modifier : component.getComponentModifiers()) {
            if (!StringUtils.equals(modifier.getRunPhase(), runPhase)) {
                continue;
            }

            boolean runCondition = true;
            if (StringUtils.isNotBlank(modifier.getRunCondition())) {
                Map<String, Object> context = new HashMap<String, Object>();
                context.putAll(component.getContext());

                if (context.isEmpty()) {
                    context.putAll(ViewLifecycle.getView().getContext());
                    context.put(UifConstants.ContextVariableNames.COMPONENT, component);
                }

                String conditionEvaluation = ViewLifecycle.getExpressionEvaluator().evaluateExpressionTemplate(context,
                        modifier.getRunCondition());
                runCondition = Boolean.parseBoolean(conditionEvaluation);
            }

            if (runCondition) {
                modifier.performModification(model, component);
            }
        }
    }
}
