/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.uif.lifecycle.model;

import java.lang.annotation.Annotation;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.krad.uif.component.ClientSideState;
import org.kuali.rice.krad.uif.component.Component;
import org.kuali.rice.krad.uif.lifecycle.ViewLifecycle;
import org.kuali.rice.krad.uif.lifecycle.ViewLifecycleTaskBase;
import org.kuali.rice.krad.uif.util.CopyUtils;
import org.kuali.rice.krad.uif.util.ObjectPropertyUtils;
import org.kuali.rice.krad.uif.view.View;
import org.kuali.rice.krad.uif.view.ViewModel;

/**
 * Synchronize client side state for the component.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class SyncClientSideStateTask extends ViewLifecycleTaskBase<Component> {

    public SyncClientSideStateTask() {
        super(Component.class);
    }

    /**
     * Updates the properties of the given component instance with the value found from the
     * corresponding map of client state (if found)
     * 
     * {@inheritDoc}
     */
    @Override
    protected void performLifecycleTask() {
        Component component = (Component) getElementState().getElement();
        ViewModel model = (ViewModel) ViewLifecycle.getModel();

        // find the map of state that was sent for component (if any)
        Map<String, Object> clientSideState = model.getClientStateForSyncing();
        if (!(component instanceof View) && clientSideState.containsKey(component.getId())) {
            @SuppressWarnings("unchecked")
            Map<String, Object> componentState =
                    (Map<String, Object>) clientSideState.get(component.getId());
            clientSideState = componentState;
        }

        // if state was sent, match with fields on the component that are annotated to have client state
        if ((clientSideState != null) && (!clientSideState.isEmpty())) {
            Map<String, Annotation> annotatedFields = CopyUtils.getFieldsWithAnnotation(component.getClass(),
                    ClientSideState.class);

            for (Entry<String, Annotation> annotatedField : annotatedFields.entrySet()) {
                ClientSideState clientSideStateAnnot = (ClientSideState) annotatedField.getValue();

                String variableName = clientSideStateAnnot.variableName();
                if (StringUtils.isBlank(variableName)) {
                    variableName = annotatedField.getKey();
                }

                if (clientSideState.containsKey(variableName)) {
                    Object value = clientSideState.get(variableName);
                    ObjectPropertyUtils.setPropertyValue(component, annotatedField.getKey(), value);
                }
            }
        }
    }

}
