/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.uif.view;

import org.kuali.rice.krad.datadictionary.parse.BeanTag;
import org.kuali.rice.krad.datadictionary.parse.BeanTagAttribute;
import org.kuali.rice.krad.uif.component.Component;
import org.kuali.rice.krad.uif.element.Message;
import org.kuali.rice.krad.uif.util.ComponentFactory;

import java.util.List;

/**
 * View that presents a message to the user (for example an application error message)
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@BeanTag(name = "messageView", parent="Uif-MessageView")
public class MessageView extends FormView {
    private static final long serialVersionUID = 5578210247236389466L;

    private Message message;

    public MessageView() {
        super();

        super.setSinglePageView(true);
    }

    /**
     * The following initialization is performed:
     *
     * <ul>
     * <li>Set the message text onto the message component and add to the page items</li>
     * </ul>
     *
     * {@inheritDoc}
     */
    @Override
    public void performInitialization(Object model) {
        super.performInitialization(model);

        List<Component> newItems = (List<Component>) getPage().getItems();
        newItems.add(message);
        getPage().setItems(newItems);
    }

    /**
     * Message component that will be used to display the message (used for styling and so on)
     *
     * @return Message component instance
     */
    @BeanTagAttribute(type = BeanTagAttribute.AttributeType.DIRECTORBYTYPE)
    public Message getMessage() {
        return message;
    }

    /**
     * Setter for the message component
     *
     * @param message
     */
    public void setMessage(Message message) {
        this.message = message;
    }

    /**
     * Message text to display in the message view.
     *
     * @return message text as string
     */
    @BeanTagAttribute
    public String getMessageText() {
        if (this.message != null) {
            return this.message.getMessageText();
        }

        return null;
    }

    /**
     * @see MessageView#getMessageText()
     */
    public void setMessageText(String messageText) {
        if (this.message == null) {
            this.message = ComponentFactory.getMessage();
        }

        this.message.setMessageText(messageText);
    }
}
