/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.web.form;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.CoreApiServiceLocator;
import org.kuali.rice.kew.api.WorkflowDocument;
import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.kim.api.services.KimApiServiceLocator;
import org.kuali.rice.krad.document.Document;
import org.kuali.rice.krad.service.KRADServiceLocatorWeb;
import org.kuali.rice.krad.web.bind.RequestAccessible;

/**
 * Base form for all <code>DocumentView</code> screens
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class DocumentFormBase extends UifFormBase {
	private static final long serialVersionUID = 2190268505427404480L;

	private static final org.apache.logging.log4j.Logger LOG = org.apache.logging.log4j.LogManager.getLogger(DocumentFormBase.class);

	private String annotation = "";

    @RequestAccessible
	private String command;

    @RequestAccessible
	private String docId;

    @RequestAccessible
	private String docTypeName;

	protected Document document;

    public DocumentFormBase() {
	    super();

	    instantiateDocument();
	}

	public String getAnnotation() {
		return this.annotation;
	}

	public void setAnnotation(String annotation) {
		this.annotation = annotation;
	}

	public Document getDocument() {
		return this.document;
	}

	public void setDocument(Document document) {
		this.document = document;
	}

    public String getDocTypeName() {
        if (this.docTypeName == null && !this.getDefaultDocumentTypeName().isEmpty()) {
            return this.getDefaultDocumentTypeName();
        }
        return this.docTypeName;
    }

	public void setDocTypeName(String docTypeName) {
		this.docTypeName = docTypeName;
	}

	public String getCommand() {
		return this.command;
	}

	public void setCommand(String command) {
		this.command = command;
	}

	public String getDocId() {
		return this.docId;
	}

	public void setDocId(String docId) {
		this.docId = docId;
	}

    protected String getDefaultDocumentTypeName() {
        return "";
    }

    protected void instantiateDocument() {
        if (document == null && StringUtils.isNotBlank(getDefaultDocumentTypeName())) {
            Class<? extends Document> documentClass = KRADServiceLocatorWeb.getDataDictionaryService()
                    .getValidDocumentClassByTypeName(getDefaultDocumentTypeName());
            try {
                Document newDocument = documentClass.newInstance();
                setDocument(newDocument);
            } catch (Exception e) {
                LOG.error("Unable to instantiate document class " + documentClass + " document type "
                        + getDefaultDocumentTypeName());
                throw new RuntimeException("Unable to instantiate document class " + documentClass + " document type "
                        + getDefaultDocumentTypeName(),e);
            }
        }
    }

	/**
	 * Retrieves the principal name (network id) for the document's initiator
	 *
	 * @return String initiator name
	 */
	public String getDocumentInitiatorNetworkId() {
		String initiatorNetworkId = "";
		if (getWorkflowDocument() != null) {
			String initiatorPrincipalId = getWorkflowDocument().getInitiatorPrincipalId();
			Person initiator = KimApiServiceLocator.getPersonService().getPerson(initiatorPrincipalId);
			if (initiator != null) {
				initiatorNetworkId = initiator.getPrincipalName();
			}
		}

		return initiatorNetworkId;
	}

	/**
	 * Retrieves the create date for the forms document and formats for
	 * presentation
	 *
	 * @return String formatted document create date
	 */
    public String getDocumentCreateDate() {
        String createDateStr = "";
        if (getWorkflowDocument() != null && getWorkflowDocument().getDateCreated() != null) {
            createDateStr = CoreApiServiceLocator.getDateTimeService().toString(
                    getWorkflowDocument().getDateCreated().toDate(), "hh:mm a MM/dd/yyyy");
        }

        return createDateStr;
    }

	/**
	 * Retrieves the <code>WorkflowDocument</code> instance from the forms
	 * document instance
	 *
	 * @return WorkflowDocument for the forms document
	 */
	public WorkflowDocument getWorkflowDocument() {
		return getDocument().getDocumentHeader().getWorkflowDocument();
	}

}
