/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.datadictionary;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.krad.datadictionary.validator.ValidationTrace;

/**
 * The defaultSort element specifies the sequence in which the lookup search results should be displayed
 *
 * <p>It contains an ascending/descending indicator and a list of attribute names.
 * JSTL: defaultSort is a Map with the following keys:
 * sortAscending (boolean String) and sortAttributes (Map).
 * By the time JSTL export occurs, the optional attributeName from the defaultSort
 * tag will have been converted into the first contained sortAttribute.
 * </p>
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class SortDefinition extends DataDictionaryDefinitionBase {
    private static final long serialVersionUID = -1092811342186612461L;

    protected boolean sortAscending = true;
    protected List<String> attributeNames = new ArrayList<String>();

    public SortDefinition() {

    }

    /**
     * The sortAttribute element defines one part of the sort key.
     * The full sort key is comprised of the sortAttribute's in the
     * order in which they have been defined.
     *
     * DD: See SortAttributesDefinition.java.
     *
     * JSTL: sortAttribute is a Map which is accessed using a
     * key of the attributeName of the sortAttribute.
     * It contains a single entry with the following key:
     * "attributeName"
     *
     * The associated value is the attributeName of the sortAttribute.
     * See LookupMapBuilder.java
     *
     * @throws IllegalArgumentException if the given attributeName is blank
     */
    public void setAttributeName(String attributeName) {
        if (StringUtils.isBlank(attributeName)) {
            throw new IllegalArgumentException("invalid (blank) attributeName");
        }
        if (!attributeNames.isEmpty()) {
            throw new IllegalStateException(
                    "unable to set sort attributeName when sortAttributes have already been added");
        }

        attributeNames.add(attributeName);
    }

    /**
     * @return the List of associated attribute names as Strings
     */
    public List<String> getAttributeNames() {
        return this.attributeNames;
    }

    /**
     * Indicates that the items must be sorted in ascending order
     *
     * @return true if items should sort in ascending order
     */
    public boolean getSortAscending() {
        return sortAscending;
    }

    /**
     * Setter for the flag to indicate ascending sorting of items
     *
     * @param sortAscending
     */
    public void setSortAscending(boolean sortAscending) {
        this.sortAscending = sortAscending;
    }

    @Override
    public void completeValidation(Class rootBusinessObjectClass, Class otherBusinessObjectClass, ValidationTrace tracer) {

        if ( attributeNames == null || attributeNames.isEmpty() ) {
            String currentValues[] = {"attributeNames = " + attributeNames, "rootBusinessObjectClass = " + rootBusinessObjectClass};
            tracer.createError("SortDefinition may not have an empty attribute list", currentValues);
        }

        for (String attributeName : attributeNames) {
            if (!DataDictionaryPropertyUtils.isPropertyOf(rootBusinessObjectClass, attributeName)) {
                String currentValues[] = {"attributeName = " + attributeName, "rootBusinessObjectClass = " + rootBusinessObjectClass};
                tracer.createError("attribute in SortDefinition not found on business object", currentValues);
            }
        }
    }

    @Override
    public String toString() {
        StringBuilder attrList = new StringBuilder("[");
        for (Iterator<String> i = attributeNames.iterator(); i.hasNext(); ) {
            attrList.append(i.next());
            if (i.hasNext()) {
                attrList.append(",");
            }
        }
        attrList.append("]");

        return "SortDefinition :  " + attrList;
    }

    /**
     * The sortAttributes element allows a multiple-part sort key to be defined
     *
     * JSTL: sortAttributes is a Map which is accessed using a key of "sortAttributes". This map contains an entry for
     * sort attribute.  The key is: attributeName of a sort field. The associated value is a sortAttribute ExportMap.
     */
    public void setAttributeNames(List<String> attributeNames) {
        this.attributeNames = attributeNames;
    }

}
