/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.document;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.persistence.Column;
import javax.persistence.Id;
import javax.persistence.MappedSuperclass;
import javax.persistence.PostLoad;
import javax.persistence.PostRemove;
import javax.persistence.PrePersist;
import javax.persistence.Transient;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.core.api.mo.common.GloballyUnique;
import org.kuali.rice.kew.api.KewApiConstants;
import org.kuali.rice.kew.api.KewApiServiceLocator;
import org.kuali.rice.kew.api.WorkflowDocument;
import org.kuali.rice.kew.api.action.ActionRequest;
import org.kuali.rice.kew.api.action.ActionType;
import org.kuali.rice.kew.api.exception.WorkflowException;
import org.kuali.rice.kew.framework.postprocessor.ActionTakenEvent;
import org.kuali.rice.kew.framework.postprocessor.DocumentRouteLevelChange;
import org.kuali.rice.kew.framework.postprocessor.DocumentRouteStatusChange;
import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.kim.api.services.KimApiServiceLocator;
import org.kuali.rice.krad.UserSessionUtils;
import org.kuali.rice.krad.bo.AdHocRoutePerson;
import org.kuali.rice.krad.bo.AdHocRouteWorkgroup;
import org.kuali.rice.krad.bo.DocumentHeader;
import org.kuali.rice.krad.bo.Note;
import org.kuali.rice.krad.bo.PersistableBusinessObjectBaseAdapter;
import org.kuali.rice.krad.datadictionary.DocumentEntry;
import org.kuali.rice.krad.datadictionary.WorkflowAttributes;
import org.kuali.rice.krad.datadictionary.WorkflowProperties;
import org.kuali.rice.krad.document.authorization.PessimisticLock;
import org.kuali.rice.krad.exception.PessimisticLockingException;
import org.kuali.rice.krad.exception.ValidationException;
import org.kuali.rice.krad.rules.rule.event.DocumentEvent;
import org.kuali.rice.krad.service.AttachmentService;
import org.kuali.rice.krad.service.DocumentSerializerService;
import org.kuali.rice.krad.service.KRADServiceLocator;
import org.kuali.rice.krad.service.KRADServiceLocatorWeb;
import org.kuali.rice.krad.service.NoteService;
import org.kuali.rice.krad.util.ErrorMessage;
import org.kuali.rice.krad.util.GlobalVariables;
import org.kuali.rice.krad.util.KRADConstants;
import org.kuali.rice.krad.util.KRADPropertyConstants;
import org.kuali.rice.krad.util.NoteType;
import org.kuali.rice.krad.util.documentserializer.AlwaysFalsePropertySerializabilityEvaluator;
import org.kuali.rice.krad.util.documentserializer.AlwaysTruePropertySerializibilityEvaluator;
import org.kuali.rice.krad.util.documentserializer.BusinessObjectPropertySerializibilityEvaluator;
import org.kuali.rice.krad.util.documentserializer.PropertySerializabilityEvaluator;
import org.kuali.rice.krad.workflow.DocumentInitiator;
import org.kuali.rice.krad.workflow.KualiDocumentXmlMaterializer;
import org.kuali.rice.krad.workflow.KualiTransactionalDocumentInformation;
import org.springframework.util.CollectionUtils;

@MappedSuperclass
public abstract class DocumentBase extends PersistableBusinessObjectBaseAdapter implements Document {
    private static final long serialVersionUID = 8530945307802647664L;
    private static final Logger LOG = LogManager.getLogger(DocumentBase.class);

    @Id
    @Column(name = "DOC_HDR_ID",length=14)
    protected String documentNumber;

    @Transient
    protected DocumentHeader documentHeader;

    @Transient
    protected List<PessimisticLock> pessimisticLocks;

    @Transient
    protected List<AdHocRoutePerson> adHocRoutePersons;
    @Transient
    protected List<AdHocRouteWorkgroup> adHocRouteWorkgroups;
    @Transient
    protected List<Note> notes;
    @Transient
    private String superUserAnnotation = "";

    private transient NoteService noteService;
    private transient AttachmentService attachmentService;

    public DocumentBase() {
        documentHeader = new DocumentHeader();
        pessimisticLocks = new ArrayList<>();
        adHocRoutePersons = new ArrayList<>();
        adHocRouteWorkgroups = new ArrayList<>();
        notes = new ArrayList<>();
    }

    @Override
    public boolean getAllowsCopy() {
        return false;
    }

    /**
     * Retrieves the title of the document
     *
     * <p>
     * This is the default document title implementation. It concatenates the document's data dictionary file label
     * attribute and
     * the document's document header description together. This title is used to populate workflow and will show up in
     * document
     * search results and user action lists.
     * </p>
     *
     * return String representing the title of the document
     *
     */
    @Override
    public String getDocumentTitle() {
        String documentTypeLabel = KewApiServiceLocator.getDocumentTypeService().getDocumentTypeByName(
                this.getDocumentHeader().getWorkflowDocument().getDocumentTypeName()).getLabel();
        if (null == documentTypeLabel) {
            documentTypeLabel = "";
        }

        String description = this.getDocumentHeader().getDocumentDescription();
        if (null == description) {
            description = "";
        }

        return documentTypeLabel + " - " + description;
    }


    @Override
    public void prepareForSave() {
        // do nothing
    }


    @Override
    public void processAfterRetrieve() {
        // do nothing
    }

    /**
     * The the default implementation for RouteLevelChange does nothing, but is meant to provide a hook for documents to
     * implement
     * for other needs.
     *
     */
    @Override
    public void doRouteLevelChange(DocumentRouteLevelChange levelChangeEvent) {
        // do nothing
    }
    
    @Override
    public void doActionTaken(ActionTakenEvent event) {
        if ((KRADServiceLocatorWeb.getDataDictionaryService().getDataDictionary().getDocumentEntry(
                this.getClass().getName()).getUseWorkflowPessimisticLocking()) && (!getNonLockingActionTakenCodes()
                .contains(event.getActionTaken().getActionTaken().getCode()))) {
            KRADServiceLocatorWeb.getPessimisticLockService().establishWorkflowPessimisticLocking(this);
        }
    }
    
    @Override
    public void afterActionTaken(ActionType performed, ActionTakenEvent event) {
        // do nothing
    }

    /**
     * Return the list of actions a user could take on a document which should not result
     * in the recalculation of the {@link PessimisticLock}s.
     * 
     * @see #doActionTaken(ActionTakenEvent)
     */
    protected List<String> getNonLockingActionTakenCodes() {
        List<String> actionTakenStatusCodes = new ArrayList<>();
        actionTakenStatusCodes.add(KewApiConstants.ACTION_TAKEN_SAVED_CD);
        actionTakenStatusCodes.add(KewApiConstants.ACTION_TAKEN_ACKNOWLEDGED_CD);
        actionTakenStatusCodes.add(KewApiConstants.ACTION_TAKEN_FYI_CD);
        actionTakenStatusCodes.add(KewApiConstants.ACTION_TAKEN_DENIED_CD);
        actionTakenStatusCodes.add(KewApiConstants.ACTION_TAKEN_CANCELED_CD);
        actionTakenStatusCodes.add(KewApiConstants.ACTION_TAKEN_LOG_DOCUMENT_ACTION_CD);
        return actionTakenStatusCodes;
    }

    /**
     * The the default implementation for afterWorkflowEngineProcess does nothing, but is meant to provide a hook for
     * documents to implement for other needs.
     *
     */
    @Override
    public void afterWorkflowEngineProcess(boolean successfullyProcessed) {
        if (KRADServiceLocatorWeb.getDataDictionaryService().getDataDictionary().getDocumentEntry(
                this.getClass().getName()).getUseWorkflowPessimisticLocking()) {
            if (successfullyProcessed) {
                KRADServiceLocatorWeb.getPessimisticLockService().releaseWorkflowPessimisticLocking(this);
            }
        }
    }

    /**
     * The the default implementation for beforeWorkflowEngineProcess does nothing, but is meant to provide a hook for
     * documents to implement for other needs.
     *
     */
    @Override
    public void beforeWorkflowEngineProcess() {
        // do nothing
    }

    /**
     * The default implementation returns no additional ids for the workflow engine to lock prior to processing.
     *
     */
    @Override
    public List<String> getWorkflowEngineDocumentIdsToLock() {
        return null;
    }

    /**
     * @see org.kuali.rice.krad.document.Copyable#toCopy()
     */
    public void toCopy() throws WorkflowException, IllegalStateException {
        if (!this.getAllowsCopy()) {
            throw new IllegalStateException(this.getClass().getName() + " does not support document-level copying");
        }
        String sourceDocumentHeaderId = getDocumentNumber();
        setNewDocumentHeader();

        //clear out notes from previous bo
        this.notes.clear();
        addCopyErrorDocumentNote("copied from document " + sourceDocumentHeaderId);
    }

    /**
     * Gets a new document header for this documents type and sets in the document instance.
     */
    protected void setNewDocumentHeader() throws WorkflowException {
        // collect the header information from the old document
        Person user = GlobalVariables.getUserSession().getPerson();
        WorkflowDocument sourceWorkflowDocument
                = KRADServiceLocatorWeb.getWorkflowDocumentService().loadWorkflowDocument(getDocumentNumber(), user);
        String sourceDocumentTypeName = sourceWorkflowDocument.getDocumentTypeName();

        // initiate the new workflow entry, get the workflow doc
        WorkflowDocument workflowDocument
                = KRADServiceLocatorWeb.getWorkflowDocumentService().createWorkflowDocument(sourceDocumentTypeName, user);
        UserSessionUtils.addWorkflowDocument(GlobalVariables.getUserSession(), workflowDocument);

        // set new values on the document header, including the document number from which it was copied
        Document newDocument = KRADServiceLocatorWeb.getDocumentService().getNewDocument(sourceDocumentTypeName);
        DocumentHeader newDocumentHeader = newDocument.getDocumentHeader();
        newDocumentHeader.setDocumentTemplateNumber(getDocumentNumber());
        newDocumentHeader.setDocumentDescription(getDocumentHeader().getDocumentDescription());
        newDocumentHeader.setOrganizationDocumentNumber(getDocumentHeader().getOrganizationDocumentNumber());

        // set the new document number on this document
        try {
            KRADServiceLocatorWeb.getLegacyDataAdapter().setObjectPropertyDeep(this,
                    KRADPropertyConstants.DOCUMENT_NUMBER, (Class<String>) documentNumber.getClass(), newDocument.getDocumentNumber());
        } catch (Exception e) {
            LOG.error("Unable to set document number property in copied document " + this, e);
            throw new RuntimeException("Unable to set document number property in copied document " + this, e);
        }

        // replace the current document header with the new document header
        setDocumentHeader(newDocument.getDocumentHeader());
    }

    /**
     * Adds a note to the document indicating it was created by a copy or error correction.
     *
     * @param noteText - text for note
     */
    protected void addCopyErrorDocumentNote(String noteText) {
        Note note;
        try {
            note = KRADServiceLocatorWeb.getDocumentService().createNoteFromDocument(this, noteText);
        } catch (Exception e) {
            logErrors();
            throw new RuntimeException("Couldn't create note on copy or error", e);
        }
        addNote(note);
    }
    
    @Override
    public String getXmlForRouteReport() {
        prepareForSave();
        populateDocumentForRouting();
        return getDocumentHeader().getWorkflowDocument().getApplicationContent();
    }
    
    @Override
    public void populateDocumentForRouting() {
        getDocumentHeader().getWorkflowDocument().setApplicationContent(serializeDocumentToXml());
    }
    
    @Override
    public String serializeDocumentToXml() {
        DocumentSerializerService documentSerializerService = KRADServiceLocatorWeb.getDocumentSerializerService();
        String xml = documentSerializerService.serializeDocumentToXmlForRouting(this);
        return xml;
    }

    /**
     * Wraps a document in an instance of KualiDocumentXmlMaterializer, that provides additional metadata for
     * serialization
     *
     */
    @Override
    public KualiDocumentXmlMaterializer wrapDocumentWithMetadataForXmlSerialization() {
        KualiTransactionalDocumentInformation transInfo = new KualiTransactionalDocumentInformation();
        DocumentInitiator initiator = new DocumentInitiator();
        String initiatorPrincipalId = getDocumentHeader().getWorkflowDocument().getDocument().getInitiatorPrincipalId();
        Person initiatorUser = KimApiServiceLocator.getPersonService().getPerson(initiatorPrincipalId);
        initiator.setPerson(initiatorUser);
        transInfo.setDocumentInitiator(initiator);
        KualiDocumentXmlMaterializer xmlWrapper = new KualiDocumentXmlMaterializer();
        xmlWrapper.setDocument(this);
        xmlWrapper.setKualiTransactionalDocumentInformation(transInfo);
        return xmlWrapper;
    }

    /**
     * If workflowProperties have been defined within the data dictionary for this document, then it returns an instance
     * of
     * {@link BusinessObjectPropertySerializibilityEvaluator} initialized with the properties.  If none have been
     * defined, then returns
     * {@link AlwaysTruePropertySerializibilityEvaluator}.
     *
     */
    @Override
    public PropertySerializabilityEvaluator getDocumentPropertySerizabilityEvaluator() {
        String docTypeName = getDocumentHeader().getWorkflowDocument().getDocumentTypeName();
        DocumentEntry documentEntry =
                KRADServiceLocatorWeb.getDataDictionaryService().getDataDictionary().getDocumentEntry(docTypeName);
        WorkflowProperties workflowProperties = documentEntry.getWorkflowProperties();
        WorkflowAttributes workflowAttributes = documentEntry.getWorkflowAttributes();
        return createPropertySerializabilityEvaluator(workflowProperties, workflowAttributes);
    }

    protected PropertySerializabilityEvaluator createPropertySerializabilityEvaluator(
            WorkflowProperties workflowProperties, WorkflowAttributes workflowAttributes) {
        if (workflowAttributes != null) {
            return new AlwaysFalsePropertySerializabilityEvaluator();
        }
        if (workflowProperties == null) {
            return new AlwaysTruePropertySerializibilityEvaluator();
        }
        PropertySerializabilityEvaluator evaluator = new BusinessObjectPropertySerializibilityEvaluator();
        evaluator.initializeEvaluatorForDocument(this);
        return evaluator;
    }

    /**
     * Returns the POJO property name of "this" document in the object returned by {@link
     * #wrapDocumentWithMetadataForXmlSerialization()}
     *
     */
    @Override
    public String getBasePathToDocumentDuringSerialization() {
        return "document";
    }
    
    @Override
    public DocumentHeader getDocumentHeader() {
        // during the transition time between OJB and JPA - the OJB hooks are not firing
        // so, we lazy load the document header.
        if ((documentHeader == null || documentHeader.getDocumentNumber() == null) && StringUtils.isNotBlank(documentNumber)) {
            documentHeader = KRADServiceLocatorWeb.getDocumentHeaderService().getDocumentHeaderById(documentNumber);
        }

        return this.documentHeader;
    }
    
    @Override
    public void setDocumentHeader(DocumentHeader documentHeader) {
        this.documentHeader = documentHeader;
    }
    
    @Override
    public String getDocumentNumber() {
        return documentNumber;
    }
    
    @Override
    public void setDocumentNumber(String documentNumber) {
        this.documentNumber = documentNumber;
    }
    
    @Override
    public List<AdHocRoutePerson> getAdHocRoutePersons() {
        return adHocRoutePersons;
    }
    
    @Override
    public void setAdHocRoutePersons(List<AdHocRoutePerson> adHocRoutePersons) {
        this.adHocRoutePersons = adHocRoutePersons;
    }
    
    @Override
    public List<AdHocRouteWorkgroup> getAdHocRouteWorkgroups() {
        return adHocRouteWorkgroups;
    }
    
    @Override
    public void setAdHocRouteWorkgroups(List<AdHocRouteWorkgroup> adHocRouteWorkgroups) {
        this.adHocRouteWorkgroups = adHocRouteWorkgroups;
    }

    /**
     * Returns null by default. Subclasses can override this to provide the node name to which any
     * adhoc requests should be attached.
     *
     * @return the name of the node to attach adhoc requests toage
     */
    @Override
    public String getAdHocRouteNodeName() {
        return null;
    }

    @Override
    public void postProcessSave(DocumentEvent event) {

    }

    /**
     * Override this method with implementation specific prepareForSave logic
     *
     */
    @Override
    public void prepareForSave(DocumentEvent event) {
        // do nothing by default
    }

    @Override
    public void validateBusinessRules(DocumentEvent event) {
        if (GlobalVariables.getMessageMap().hasErrors()) {
            logErrors();
            throw new ValidationException("errors occurred before business rule");
        }

        // perform validation against rules engine
        LOG.info("invoking rules engine on document " + getDocumentNumber());

        boolean isValid = KRADServiceLocatorWeb.getKualiRuleService().applyRules(event);

        // check to see if the br eval passed or failed
        if (!isValid) {
            logErrors();
            throw new ValidationException("business rule evaluation failed");
        } else if (GlobalVariables.getMessageMap().hasErrors()) {
            logErrors();
            throw new ValidationException(
                    "Unreported errors occurred during business rule evaluation (rule developer needs to put meaningful error messages into global ErrorMap)");
        }
        LOG.debug("validation completed");

    }

    /**
     * This method logs errors.
     */
    protected void logErrors() {
        if (LOG.isInfoEnabled()) {
            if (GlobalVariables.getMessageMap().hasErrors()) {

                for (Map.Entry<String, List<ErrorMessage>> e : GlobalVariables.getMessageMap().getAllPropertiesAndErrors()) {
                    StringBuilder logMessage = new StringBuilder();
                    logMessage.append("[").append(e.getKey()).append("] ");
                    boolean first = true;

                    List<ErrorMessage> errorList = e.getValue();
                    for (ErrorMessage em : errorList) {
                        if (first) {
                            first = false;
                        } else {
                            logMessage.append(";");
                        }
                        logMessage.append(em);
                    }

                    LOG.info(logMessage);
                }
            }
        }
    }
    
    @Override
    public List<DocumentEvent> generateSaveEvents() {
        return new ArrayList<>();
    }
    
    @Override
    public void doRouteStatusChange(DocumentRouteStatusChange statusChangeEvent) {
        // do nothing
    }

    /**
     * Returns the business object with which notes related to this document should be associated.
     * By default, the {@link DocumentHeader} of this document will be returned as the note target.
     *
     * <p>Sub classes can override this method if they want notes to be associated with something
     * other than the document header.  If this method is overridden, the {@link #getNoteType()}
     * method should be overridden to return {@link NoteType#BUSINESS_OBJECT}
     *
     * @return Returns the documentBusinessObject.
     */
    @Override
    public GloballyUnique getNoteTarget() {
        return getDocumentHeader();
    }

    /**
     * Returns the {@link NoteType} to use for notes associated with this document.
     * By default this returns {@link NoteType#DOCUMENT_HEADER} since notes are
     * associated with the {@link DocumentHeader} record by default.
     *
     * <p>The case in which this should be overridden is if {@link #getNoteTarget()} is
     * overridden to return an object other than the DocumentHeader.
     *
     * @return the note type to use for notes associated with this document
     */
    @Override
    public NoteType getNoteType() {
        return NoteType.DOCUMENT_HEADER;
    }
    
    @Override
    public void addNote(Note note) {
        if (note == null) {
            throw new IllegalArgumentException("Note cannot be null.");
        }
        notes.add(note);
    }
    
    @Override
    public boolean removeNote(Note note) {
        if (note == null) {
            throw new IllegalArgumentException("Note cannot be null.");
        }
        return notes.remove(note);
    }
    
    @Override
    public Note getNote(int index) {
        return notes.get(index);
    }
    
    @Override
    public List<Note> getNotes() {
        if (CollectionUtils.isEmpty(notes) && getNoteType().equals(NoteType.BUSINESS_OBJECT) && StringUtils.isNotBlank(
                getNoteTarget().getObjectId())) {
            notes = getNoteService().getByRemoteObjectId(getNoteTarget().getObjectId());
        }

        return notes;
    }
    
    @Override
    public void setNotes(List<Note> notes) {
        if (notes == null) {
            throw new IllegalArgumentException("List of notes must be non-null.");
        }
        this.notes = notes;
    }
    
    @Override
    public List<ActionRequest> getActionRequests() {
        return KewApiServiceLocator.getWorkflowDocumentService().getPendingActionRequests(getDocumentNumber());
    }


    @Override
    public String getSuperUserAnnotation() {
        return superUserAnnotation;
    }


    @Override
    public void setSuperUserAnnotation(String superUserAnnotation) {
        this.superUserAnnotation = superUserAnnotation;
    }

    /**
     * Loads the KRAD document header via the document header service.
     */
    @PostLoad
    protected void postLoad() {
        documentHeader = KRADServiceLocatorWeb.getDocumentHeaderService().getDocumentHeaderById(documentNumber);
        refreshPessimisticLocks();
    }

    /**
     * Save the KRAD document header via the document header service.
     */
    @Override
    @PrePersist
    protected void prePersist() {
        super.prePersist();
        // KRAD/JPA - have to change the handle to object to that just saved
        documentHeader = KRADServiceLocatorWeb.getDocumentHeaderService().saveDocumentHeader(documentHeader);
    }

    /**
     * This overridden method is used to delete the {@link DocumentHeader} object due to the system not being able to
     * manage the {@link DocumentHeader} object via mapping files
     *
     */
    @PostRemove
    protected void postRemove() {
        KRADServiceLocatorWeb.getDocumentHeaderService().deleteDocumentHeader(getDocumentHeader());
    }
    
    @Override
    public List<PessimisticLock> getPessimisticLocks() {
        return pessimisticLocks;
    }
    
    @Override
    public void refreshPessimisticLocks() {
        pessimisticLocks = KRADServiceLocatorWeb.getPessimisticLockService().getPessimisticLocksForDocument(documentNumber);
    }
    
    public void setPessimisticLocks(List<PessimisticLock> pessimisticLocks) {
        this.pessimisticLocks = pessimisticLocks;
    }
    
    @Override
    public void addPessimisticLock(PessimisticLock lock) {
        pessimisticLocks.add(lock);
    }
    
    @Override
    @Deprecated
    public List<String> getLockClearingMethodNames() {
        return getLockClearningMethodNames();
    }
    
    @Override
    @Deprecated
    public List<String> getLockClearningMethodNames() {
        List<String> methodToCalls = new ArrayList<String>();
        methodToCalls.add(KRADConstants.CLOSE_METHOD);
        methodToCalls.add(KRADConstants.CANCEL_METHOD);
        methodToCalls.add(KRADConstants.ROUTE_METHOD);
        methodToCalls.add(KRADConstants.APPROVE_METHOD);
        methodToCalls.add(KRADConstants.DISAPPROVE_METHOD);
        methodToCalls.add(KRADConstants.ACKNOWLEDGE_METHOD);
        return methodToCalls;
    }

    /**
     * This default implementation simply returns false to indicate that custom lock descriptors are not supported by
     * DocumentBase. If custom lock
     * descriptors are needed, the appropriate subclasses should override this method.
     *
     */
    @Override
    public boolean useCustomLockDescriptors() {
        return false;
    }

    /**
     * This default implementation just throws a PessimisticLockingException. Subclasses of DocumentBase that need
     * support for custom lock descriptors
     * should override this method.
     *
     */
    @Override
    public String getCustomLockDescriptor(Person user) {
        throw new PessimisticLockingException("Document " + getDocumentNumber() +
                " is using pessimistic locking with custom lock descriptors, but the document class has not overridden the getCustomLockDescriptor method");
    }

    protected AttachmentService getAttachmentService() {
        if (attachmentService == null) {
            attachmentService = KRADServiceLocator.getAttachmentService();
        }
        return attachmentService;
    }

    protected NoteService getNoteService() {
        if (noteService == null) {
            noteService = KRADServiceLocator.getNoteService();
        }
        return noteService;
    }

    /**
     * Overrides this OJB method to accept the no-longer-bound documentHeader reference
     * and perform the refresh via services instead of via OJB.
     *
     * For any other property, it works as before.
     *
     * @deprecated This is a KNS/OJB-related method.  It should not be used on KRAD/JPA-based documents.
     */
    @Override
    @Deprecated
    public void refreshReferenceObject(String referenceObjectName) {
        if ( StringUtils.equals( referenceObjectName, "documentHeader" ) ) {
            documentHeader = KRADServiceLocatorWeb.getDocumentHeaderService().getDocumentHeaderById(documentNumber);
        } else {
            super.refreshReferenceObject(referenceObjectName);
        }
    }
}
