/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.inquiry;

import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.krad.uif.widget.Inquiry;

import java.util.Map;

/**
 * Provides the contract for implementing an inquiry within the
 * inquiry framework
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public interface Inquirable {

    /**
     * Class for the data object the inquirable should retrieve
     *
     * <p>
     * Must be set before invoking any other operations on the <code>Inquirable</code>,
     * including the retrieveDataObject method
     * </p>
     *
     * @return inquiry data object class
     */
    Class<?> getDataObjectClass();

    /**
     * @see Inquirable#getDataObjectClass()
     */
    void setDataObjectClass(Class<?> dataObjectClass);

    /**
     * Responsible for retrieving the data object from its data source
     * (database, service call, etc) based on the given map of field
     * name/value pairs
     *
     * <p>
     * Given map can contain more than fields (primary key or other) necessary
     * for retrieving the data object. Method will use the fields necessary
     * based on the metadata for the data object class configured on the inquirable
     * </p>
     *
     * @param fieldValues - a map of string field names and values
     * @return the data object or null if not found
     */
    Object retrieveDataObject(Map<String, String> fieldValues);

    /**
     * Invoked by the <code>ViewHelperService</code> to build a link to the
     * inquiry
     *
     * <p>
     * Note this is used primarily for custom <code>Inquirable</code>
     * implementations to customize the inquiry class or parameters for an
     * inquiry. Instead of building the full inquiry link, implementations can
     * make a callback to
     * org.kuali.rice.krad.uif.widget.Inquiry.buildInquiryLink(Object, String,
     * Class<?>, Map<String, String>) given an inquiry class and parameters to
     * build the link field.
     * </p>
     *
     * @param dataObject - parent object for the inquiry property
     * @param propertyName - name of the property the inquiry is being built for
     * @param inquiry - instance of the inquiry widget being built for the property
     */
    void buildInquirableLink(Object dataObject, String propertyName, Inquiry inquiry);

    /**
     * Checks appropriate permissions to determine whether the given user is
     * authorized to download note attachments associated with this inquiry
     * data object. Should probably be in an authorizer class, but since
     * inquiries don't always have document or view associations this is the
     * next best place.
     *
     * @param user - the user for whom to check authorization
     * @return whether or not the user is authorized to download note attachments
     */
    boolean canViewInquiryNoteAttachment(Person user);
}
