/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.uif.field;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.krad.datadictionary.parse.BeanTag;
import org.kuali.rice.krad.datadictionary.parse.BeanTagAttribute;
import org.kuali.rice.krad.uif.component.Component;
import org.kuali.rice.krad.uif.element.Header;
import org.kuali.rice.krad.uif.element.Image;
import org.kuali.rice.krad.uif.element.Message;
import org.kuali.rice.krad.uif.util.LifecycleElement;

/**
 * Field that wraps an image content element.
 *
 * <p>
 * Puts a <code>&lt;DIV&gt;</code> tag around an image element. This allows for labeling, styling, etc.
 * </p>
 *
 * @see org.kuali.rice.krad.uif.element.Image
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@BeanTag(name = "imageField", parent = "Uif-ImageField")
public class ImageField extends FieldBase {
    private static final long serialVersionUID = -7994212503770623408L;

    private Image image;

    public ImageField() {
        super();
    }

    /**
     * PerformFinalize override - calls super, corrects the field's Label for attribute to point to this field's
     * content
     *
     * @param model the model
     * @param parent the parent component
     */
    @Override
    public void performFinalize(Object model, LifecycleElement parent) {
        super.performFinalize(model, parent);

        //determine what id to use for the for attribute of the label, if present
        if (this.getFieldLabel() != null && this.getImage() != null && StringUtils.isNotBlank(this.getImage().getId())) {
            this.getFieldLabel().setLabelForComponentId(this.getImage().getId());
        }
    }

    /**
     * Retrieves the {@link Image} element wrapped by this field
     *
     * @return the Image element representing the HTML IMG element
     */
    @BeanTagAttribute(type= BeanTagAttribute.AttributeType.DIRECTORBYTYPE)
    public Image getImage() {
        return image;
    }

    /**
     * Sets the Image to be wrapped by this field
     *
     * @param image the Image element to be wrapped by this field
     */
    public void setImage(Image image) {
        this.image = image;
    }

    /**
     * Retrieves the URL the image wrapped by this field
     *
     * @see org.kuali.rice.krad.uif.element.Image#getSource()
     * @return the URL for the image
     */
    @BeanTagAttribute
    public String getSource() {
        return image.getSource();
    }

    /**
     * Sets the source URL for the Image associated with this field
     *
     * @param source URL for the image
     */
    public void setSource(String source) {
        image.setSource(source);
    }

    /**
     * Provides alternate information for the image element
     *
     * <p>The altText property specifies an alternate text for an image. It is displayed by the browser
     * if the image cannot be displayed.  This is especially important for accessibility, because screen
     * readers can't understand images, but rather will read aloud the alternative text assigned to them.
     * </p>
     *
     * @see org.kuali.rice.krad.uif.element.Image#getAltText()
     * @return alternative information about this image
     */
    @BeanTagAttribute
    public String getAltText() {
        return image.getAltText();
    }

    /**
     * Sets the alternate text attribute of the image assosiated with this field
     *
     * @param altText the alternative information about the image
     */
    public void setAltText(String altText) {
        image.setAltText(altText);
    }

    /**
     * Gets the height of the image
     *
     * @return height
     */
    @BeanTagAttribute
    public String getHeight() {
        return image.getHeight();
    }

    /**
     * Sets the height of the image
     *
     * @param height
     */
    public void setHeight(String height) {
        image.setHeight(height);
    }

    /**
     * Gets the width of the image
     *
     * @return width
     */
    @Override
    @BeanTagAttribute
    public String getWidth() {
        return image.getWidth();
    }

    /**
     * Sets the width of the image
     *
     * @param width
     */
    @Override
    public void setWidth(String width) {
        if (image != null) {
            image.setWidth(width);
        }
    }

    /**
     * Gets the caption header text
     *
     * @return captionHeaderText
     */
    @BeanTagAttribute
    public String getCaptionHeaderText() {
        return image.getCaptionHeaderText();
    }

    /**
     * Sets the caption header text
     *
     * @param captionHeaderText
     */
    public void setCaptionHeaderText(String captionHeaderText) {
        image.setCaptionHeaderText(captionHeaderText);
    }

    /**
     * Gets the caption header
     *
     * @return captionHeader
     */
    @BeanTagAttribute
    public Header getCaptionHeader() {
        return image.getCaptionHeader();
    }

    /**
     * Sets the caption header
     *
     * @param captionHeader
     */
    public void setCaptionHeader(Header captionHeader) {
        image.setCaptionHeader(captionHeader);
    }

    /**
     * Gets the cutline text
     *
     * @return cutlineText
     */
    @BeanTagAttribute
    public String getCutlineText() {
        return image.getCutlineText();
    }

    /**
     * Sets the cutline text
     *
     * @param cutlineText
     */
    public void setCutlineText(String cutlineText) {
        image.setCutlineText(cutlineText);
    }

    /**
     * Gets the cutline
     *
     * @return cutline
     */
    @BeanTagAttribute
    public Message getCutline() {
        return image.getCutlineMessage();
    }

    /**
     * Sets the cutline
     *
     * @param cutline
     */
    public void setCutline(Message cutline) {
        image.setCutlineMessage(cutline);
    }

    /**
     * Gets boolen of whether the caption header is above the image
     *
     * @return captionHeaderAboveImage
     */
    @BeanTagAttribute
    public boolean isCaptionHeaderAboveImage() {
        return image.isCaptionHeaderPlacementAboveImage();
    }

    /**
     * Sets boolen of whether the caption header is above the image
     *
     * @param captionHeaderAboveImage
     */
    public void setCaptionHeaderAboveImage(boolean captionHeaderAboveImage) {
        image.setCaptionHeaderPlacementAboveImage(captionHeaderAboveImage);
    }
}
