/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.util.documentserializer;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.krad.datadictionary.DataDictionary;
import org.kuali.rice.krad.datadictionary.DocumentEntry;
import org.kuali.rice.krad.datadictionary.WorkflowProperties;
import org.kuali.rice.krad.datadictionary.WorkflowProperty;
import org.kuali.rice.krad.datadictionary.WorkflowPropertyGroup;
import org.kuali.rice.krad.document.Document;
import org.kuali.rice.krad.service.KRADServiceLocatorWeb;

import java.util.List;

/**
 * This implementation of {@link PropertySerializabilityEvaluator} uses the &lt;workflowProperties&gt; defined within the data dictionary
 * for a document.  If the property being serialized corresponds to one of the properties in the data dictionary, then it will be serialized.
 * If a property specified in the data dictionary corresponds to a business object, then all primitives will be serialized of the business object.
 * All primitives of a primitive that has already been serialized will be serialized as well.   If a property specified in the data dictionary corresponds
 * to a collection, then all primitives of all collection elements will be serialized.
 *
 */
public class BusinessObjectPropertySerializibilityEvaluator extends PropertySerializabilityEvaluatorBase implements PropertySerializabilityEvaluator {

    /**
     * Reads the data dictionary to determine which properties of the document should be serialized.
     *
     * {@inheritDoc}
     */
	@Override
    public void initializeEvaluatorForDocument(Document document) {
        DataDictionary dictionary = KRADServiceLocatorWeb.getDataDictionaryService().getDataDictionary();
        DocumentEntry docEntry = dictionary.getDocumentEntry(document.getDocumentHeader().getWorkflowDocument().getDocumentTypeName());
        WorkflowProperties workflowProperties = docEntry.getWorkflowProperties();
        List<WorkflowPropertyGroup> groups = workflowProperties.getWorkflowPropertyGroups();

        serializableProperties = new PropertySerializerTrie();

        for (WorkflowPropertyGroup group : groups) {
            // the basepath of each workflow property group is serializable
            if (StringUtils.isEmpty(group.getBasePath())) {
                // automatically serialize all primitives of document when the base path is null or empty string
                serializableProperties.addSerializablePropertyName(document.getBasePathToDocumentDuringSerialization(), false);
            }
            else {
               serializableProperties.addSerializablePropertyName(group.getBasePath(), false);
            }

            for (WorkflowProperty property : group.getWorkflowProperties()) {
                String fullPath;
                if (StringUtils.isEmpty(group.getBasePath())) {
                    fullPath = document.getBasePathToDocumentDuringSerialization() + "." + property.getPath();
                }
                else {
                    fullPath = group.getBasePath() + "." + property.getPath();
                }
                serializableProperties.addSerializablePropertyName(fullPath, false);
            }
        }
    }

}
