/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.web.controller;

import org.kuali.rice.krad.service.KRADServiceLocatorWeb;
import org.kuali.rice.krad.uif.UifConstants;
import org.kuali.rice.krad.uif.util.ScriptUtils;
import org.kuali.rice.krad.util.GlobalVariables;
import org.kuali.rice.krad.util.KRADPropertyConstants;
import org.kuali.rice.krad.web.form.UifFormBase;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * Controller that receives various ajax requests from the client to manager server side state
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@Controller
@RequestMapping(value = "/listener")
public class UifClientListener extends UifControllerBase {

    /**
     * {@inheritDoc}
     */
    @Override
    protected UifFormBase createInitialForm() {
        return new UifFormBase();
    }

    /**
     * Invoked from the client when the user is leaving a view (by the portal tabs or other mechanism) to clear
     * the form from session storage
     *
     * @param formKeyToClear key of form that should be cleared
     * @return String json success string
     */
    @MethodAccessible
    @RequestMapping(params = "methodToCall=clearForm")
    public
    @ResponseBody
    String clearForm(@RequestParam("formKeyToClear") String formKeyToClear, HttpServletRequest request,
            HttpServletResponse response) {

        // clear form from session
        GlobalVariables.getUifFormManager().removeFormWithHistoryFormsByKey(formKeyToClear);

        return "{\"status\":\"success\"}";
    }

    /**
     * Invoked from the client to retrieve text for a message
     *
     * @param key - key for the message
     * @return String response in JSON format containing the message text
     */
    @MethodAccessible
    @RequestMapping(params = "methodToCall=retrieveMessage")
    public
    @ResponseBody
    String retrieveMessage(@RequestParam("key") String key, HttpServletRequest request, HttpServletResponse response) {
        // namespace and component are not required, therefore may be null
        String namespace = request.getParameter(KRADPropertyConstants.NAMESPACE_CODE);
        String componentCode = request.getParameter(KRADPropertyConstants.COMPONENT_CODE);

        String messageText = KRADServiceLocatorWeb.getMessageService().getMessageText(namespace, componentCode, key);

        if (messageText == null) {
            messageText = "";
        }
        else {
            messageText = ScriptUtils.escapeJSONString(messageText);
        }

        return "{\"messageText\":\"" + messageText + "\"}";
    }

    /**
     * Invoked from the session timeout warning dialog to keep a session alive on behalf of a user
     *
     * @return String json success string
     */
    @MethodAccessible
    @RequestMapping(params = "methodToCall=keepSessionAlive")
    public
    @ResponseBody
    String keepSessionAlive(HttpServletRequest request, HttpServletResponse response) {
        return "{\"status\":\"success\"}";
    }

    /**
     * Invoked from the session timeout warning dialog to log the user out, forwards to logout message view
     */
    @MethodAccessible
    @RequestMapping(params = "methodToCall=logout")
    public ModelAndView logout(@ModelAttribute("KualiForm") UifFormBase form, BindingResult result,
                HttpServletRequest request, HttpServletResponse response) {

        request.getSession().invalidate();

        return getModelAndViewWithInit(form, UifConstants.LOGGED_OUT_VIEW_ID);
    }

}
