/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krms.impl.repository;

import org.kuali.rice.krms.api.repository.action.ActionDefinition;
import org.kuali.rice.krms.api.repository.agenda.AgendaItemDefinition;
import org.kuali.rice.krms.api.repository.proposition.PropositionDefinition;
import org.kuali.rice.krms.api.repository.rule.RuleDefinition;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;

/**
 * This is the interface for accessing KRMS repository Rule related bos 
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 *
 */
public interface RuleBoService {
    /**
     * This will create a {@link RuleDefinition} exactly like the parameter passed in.
     *
     * @param rule  The Rule to create
     * @throws IllegalArgumentException if the rule is null
     * @throws IllegalStateException if the rule already exists in the system
     */
    @CacheEvict(value={RuleDefinition.Cache.NAME, PropositionDefinition.Cache.NAME, ActionDefinition.Cache.NAME, AgendaItemDefinition.Cache.NAME}, allEntries = true)
    public RuleDefinition createRule(RuleDefinition rule);

    /**
     * This will update an existing {@link RuleDefinition}.
     *
     * @param rule  The Rule to update
     * @throws IllegalArgumentException if the Rule is null
     * @throws IllegalStateException if the Rule does not exists in the system
     */
    @CacheEvict(value={RuleDefinition.Cache.NAME, PropositionDefinition.Cache.NAME, ActionDefinition.Cache.NAME, AgendaItemDefinition.Cache.NAME}, allEntries = true)
    public RuleDefinition updateRule(RuleDefinition rule);

    /**
     * Delete the {@link RuleDefinition} with the given id.
     *
     * @param ruleId to delete.
     * @throws IllegalArgumentException if the Rule is null.
     * @throws IllegalStateException if the Rule does not exists in the system
     *
     */
    public void deleteRule(String ruleId);

    /**
     * Retrieves an Rule from the repository based on the given rule id.
     *
     * @param ruleId the id of the Rule to retrieve
     * @return an {@link RuleDefinition} identified by the given actionId.
     * A null reference is returned if an invalid or non-existent id is supplied.
     * @throws IllegalArgumentException if the ruleId is null or blank.
     */
    @Cacheable(value= RuleDefinition.Cache.NAME, key="'ruleId=' + #p0")
    public RuleDefinition getRuleByRuleId(String ruleId);

    /**
     * Retrieves an Rule from the repository based on the provided rule name
     * and namespace.
     *
     * @param name the name of the Rule to retrieve.
     * @param namespace the namespace that the rule is under.
     * @return an {@link RuleDefinition} identified by the given name and namespace.
     * A null reference is returned if an invalid or non-existent name and
     * namespace combination is supplied.
     * @throws IllegalArgumentException if the either the name or the namespace
     * is null or blank.
     */
    @Cacheable(value= RuleDefinition.Cache.NAME, key="'name=' + #p0 + '|' + 'namespace=' + #p1")
    public RuleDefinition getRuleByNameAndNamespace(String name, String namespace);

}
