/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.messaging.servicehandlers;

import java.io.IOException;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.xml.namespace.QName;

import org.apache.wss4j.common.ext.WSPasswordCallback;
import org.kuali.rice.core.api.exception.RiceRuntimeException;
import org.kuali.rice.ksb.service.KSBServiceLocator;

/**
 * CallbackHandler that verifies the password and username is correct for a service
 * secured with basic authentication.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class BasicAuthenticationHandler implements CallbackHandler {

    private String serviceNameSpaceURI;
    private QName localServiceName;

    /**
     * Initialize the BasicAuthenticationHandler with the serviceNameSpaceURI and localServiceName
     *
     * @param serviceNameSpaceURI the serviceNameSpaceURI to use
     * @param serviceName the serviceName to use
     */
    public BasicAuthenticationHandler(String serviceNameSpaceURI, QName serviceName) {
        this.serviceNameSpaceURI = serviceNameSpaceURI;
        this.localServiceName = serviceName;
    }

    /**
     * @param callbacks an array of Callback objects
     * @throws RiceRuntimeException if the username or password is invalid
     * @see javax.security.auth.callback.CallbackHandler#handle(javax.security.auth.callback.Callback[])
     */
    @Override
    public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException {
        if (callbacks[0] != null && callbacks[0] instanceof WSPasswordCallback) {
            WSPasswordCallback pc = (WSPasswordCallback) callbacks[0];
            String password = KSBServiceLocator.getBasicAuthenticationService().getPasswordForService(this.serviceNameSpaceURI,
                    this.localServiceName, pc.getIdentifier());
            pc.setPassword(password);
        }
    }
}
