/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.security.httpinvoker;

import org.apache.commons.codec.binary.Base64;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpPost;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.rice.core.api.security.credentials.CredentialsSource;
import org.kuali.rice.ksb.api.bus.ServiceConfiguration;
import org.kuali.rice.ksb.messaging.KSBHttpInvokerRequestExecutor;
import org.kuali.rice.ksb.security.credentials.UsernamePasswordCredentials;
import org.springframework.remoting.httpinvoker.HttpInvokerClientConfiguration;
import org.springframework.util.Assert;

import java.io.ByteArrayOutputStream;
import java.io.IOException;


/**
 * Extension to {@link KSBHttpInvokerRequestExecutor} that retrieves
 * credentials from the CredentialsSource and places them in a BASIC HTTP
 * Authorization header.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 * @since 0.9
 */
public final class AuthenticationCommonsHttpInvokerRequestExecutor extends
    KSBHttpInvokerRequestExecutor {

    private static final Logger LOG = LogManager.getLogger(AuthenticationCommonsHttpInvokerRequestExecutor.class);

    /**
     * Source of the credentials to pass via BASIC AUTH.
     */
    private final CredentialsSource credentialsSource;

    /**
     * Details about the service that the CredentialsSource may need.
     */
    private final ServiceConfiguration serviceConfiguration;

    /**
     * Constructor that accepts the CredentialsSource and Service Info.
     *
     * @param httpClient the http client
     * @param credentialsSource the source of credentials.
     * @param serviceConfiguration the service configuration.
     */
    public AuthenticationCommonsHttpInvokerRequestExecutor(final HttpClient httpClient,
        final CredentialsSource credentialsSource, final ServiceConfiguration serviceConfiguration) {
        super(httpClient);
        Assert.notNull(credentialsSource, "credentialsSource cannot be null.");
        Assert.notNull(serviceConfiguration, "serviceConfiguration cannot be null.");
        this.credentialsSource = credentialsSource;
        this.serviceConfiguration = serviceConfiguration;
    }

    /**
     * Overridden to obtain the Credentials from the CredentialsSource and pass
     * them via HTTP BASIC Authorization.
     */

    @Override
    protected void setRequestBody(final HttpInvokerClientConfiguration config,
                                  final HttpPost httpPost, final ByteArrayOutputStream baos) throws IOException {
    	final UsernamePasswordCredentials credentials = (UsernamePasswordCredentials) this.credentialsSource.getCredentials(this.serviceConfiguration.getEndpointUrl().toExternalForm());

        final String base64 = credentials.getUsername() + ":"
        + credentials.getPassword();
        httpPost.addHeader("Authorization", "Basic " + new String(Base64.encodeBase64(base64.getBytes())));

        if (LOG.isDebugEnabled()) {
            LOG
                .debug("HttpInvocation now presenting via BASIC authentication CredentialsSource-derived: "
                    + credentials.getUsername());
        }
        
        super.setRequestBody(config, httpPost, baos);
    }
}
