/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.service.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.ksb.service.BasicAuthenticationConnectionCredentials;
import org.kuali.rice.ksb.service.BasicAuthenticationCredentials;
import org.kuali.rice.ksb.service.BasicAuthenticationService;

/**
 * Implements the BasicAuthenticationService
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 * @see org.kuali.rice.ksb.service.BasicAuthenticationService
 */

public class BasicAuthenticationServiceImpl implements BasicAuthenticationService {

    private Map<QName, List<BasicAuthenticationCredentials>> serviceCredentialsMap;
    private Map<QName, BasicAuthenticationConnectionCredentials> connectionCredentialsMap;

    /**
     * Constructs BasicAuthenticationServiceImpl with a serviceCredentialsMap and a connectionCredentialsMap
     */
    public BasicAuthenticationServiceImpl() {
        this.serviceCredentialsMap = Collections.synchronizedMap(
                new HashMap<QName, List<BasicAuthenticationCredentials>>());
        this.connectionCredentialsMap = Collections.synchronizedMap(
                new HashMap<QName, BasicAuthenticationConnectionCredentials>());
    }

    @Override
    public boolean checkServiceAuthentication(String serviceNameSpaceURI, QName serviceName, String username,
                                              String password) {
        List<BasicAuthenticationCredentials> credentialsList = serviceCredentialsMap.get(serviceName);
        if (credentialsList != null) {
            synchronized (credentialsList) {
                for (BasicAuthenticationCredentials credentials : credentialsList) {
                    if (StringUtils.equals(username, credentials.getUsername()) && StringUtils.equals(
                            serviceNameSpaceURI, credentials.getServiceNameSpaceURI())) {
                        return StringUtils.equals(password, credentials.getPassword());
                    }
                }
            }
        }
        return false;
    }

    @Override
    public String getPasswordForService(String serviceNameSpaceURI, QName serviceName, String username) {
        List<BasicAuthenticationCredentials> credentialsList = serviceCredentialsMap.get(serviceName);
        if (credentialsList != null) {
            synchronized (credentialsList) {
                for (BasicAuthenticationCredentials credentials : credentialsList) {
                    if (StringUtils.equals(username, credentials.getUsername()) && StringUtils.equals(
                            serviceNameSpaceURI, credentials.getServiceNameSpaceURI())) {
                        return credentials.getPassword();
                    }
                }
            }
        }
        return null;
    }

    @Override
    public BasicAuthenticationConnectionCredentials getConnectionCredentials(String serviceNameSpaceURI,
                                                                             String serviceName) {
        return connectionCredentialsMap.get(new QName(serviceNameSpaceURI, serviceName));
    }

    @Override
    public void registerServiceCredentials(BasicAuthenticationCredentials credentials) {
        synchronized (serviceCredentialsMap) {
            QName serviceName = new QName(credentials.getServiceNameSpaceURI(), credentials.getLocalServiceName());
            List<BasicAuthenticationCredentials> credentialsList = serviceCredentialsMap.get(serviceName);
            if (credentialsList == null) {
                credentialsList = Collections.synchronizedList(new ArrayList<BasicAuthenticationCredentials>());
                serviceCredentialsMap.put(serviceName, credentialsList);
            }
            credentialsList.add(credentials);
        }
    }

    @Override
    public void registerConnectionCredentials(BasicAuthenticationConnectionCredentials credentials) {
        synchronized (connectionCredentialsMap) {
            connectionCredentialsMap.put(new QName(credentials.getServiceNameSpaceURI(),
                    credentials.getLocalServiceName()), credentials);
        }
    }

}
