/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.messaging.serviceconnectors;

import java.net.URL;

import org.kuali.rice.core.api.security.credentials.CredentialsSource;
import org.kuali.rice.ksb.api.bus.ServiceConfiguration;
import org.kuali.rice.ksb.messaging.BusClientFailureProxy;
import org.kuali.rice.ksb.messaging.bam.BAMClientProxy;
import org.springframework.util.Assert;


/**
 * Abstract implementation of the ServiceConnector that holds the ServiceInfo
 * and the CredentialsSource as well as providing convenience proxy methods.
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 * @since 0.9
 * 
 */
public abstract class AbstractServiceConnector implements ServiceConnector {

	/**
	 * Maintains the information about the service.  This should never be null.
	 */
	private ServiceConfiguration serviceConfiguration;
	private URL alternateEndpointUrl;

	/**
	 * Maintains the credentials needed by the service.  This may be null.
	 */
	private CredentialsSource credentialsSource;

	public AbstractServiceConnector(final ServiceConfiguration serviceConfiguration) {
		this(serviceConfiguration, null);
	}
	
	public AbstractServiceConnector(final ServiceConfiguration serviceConfiguration, URL alternateEndpointUrl) {
		Assert.notNull(serviceConfiguration, "serviceConfiguration cannot be null");
		this.serviceConfiguration = serviceConfiguration;
		this.alternateEndpointUrl = alternateEndpointUrl;
	}
	
	public URL getActualEndpointUrl() {
		if (alternateEndpointUrl != null) {
            return alternateEndpointUrl;
        }
        return getServiceConfiguration().getEndpointUrl();
	}

	public ServiceConfiguration getServiceConfiguration() {
		return this.serviceConfiguration;
	}

	@Override
    public void setCredentialsSource(final CredentialsSource credentialsSource) {
		this.credentialsSource = credentialsSource;
	}

	protected CredentialsSource getCredentialsSource() {
		return this.credentialsSource;
	}

	protected Object getServiceProxyWithFailureMode(final Object service,
			final ServiceConfiguration serviceConfiguration) {
		Object bamWrappedClientProxy = BAMClientProxy.wrap(service, serviceConfiguration);
        if (!serviceConfiguration.isQueue()) {
            return bamWrappedClientProxy;
        }
		return BusClientFailureProxy.wrap(bamWrappedClientProxy, serviceConfiguration);
	}
}
