/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.messaging.serviceexporters;

import java.util.List;

import org.apache.cxf.Bus;
import org.apache.cxf.endpoint.Server;
import org.apache.cxf.endpoint.ServerRegistry;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.ksb.api.bus.ServiceDefinition;
import org.kuali.rice.ksb.messaging.bam.BAMServerProxy;
import org.kuali.rice.ksb.messaging.servlet.CXFServletControllerAdapter;

/**
 * Abstract ServiceExporter for web services 
 * 
 * @author Kuali Rice Team (rice.collab@kuali.org)
 *
 */
public abstract class AbstractWebServiceExporter implements ServiceExporter {

    static final Logger LOG = LogManager.getLogger(AbstractWebServiceExporter.class);
    
    private final ServiceDefinition serviceDefinition;
    private final Bus cxfBus;

    protected abstract void publishService(ServiceDefinition serviceDefinition, Object serviceImpl, String address) throws Exception;

    public AbstractWebServiceExporter(ServiceDefinition serviceDefinition, Bus cxfBus) {
        this.serviceDefinition = serviceDefinition;
        this.cxfBus = cxfBus;
    }
    
    @Override
    public Object exportService(ServiceDefinition serviceDefinition) {
    	try {			
    		String serviceAddress = getServiceAddress(serviceDefinition);
    		
    		//Publish the CXF service if it hasn't already been published
    		if (!(isServicePublished(serviceAddress))){
    			publishService(serviceDefinition, serviceDefinition.getService(), serviceAddress);
    		}
    		
    		//Create a CXF mvc controller for this service
    		CXFServletControllerAdapter cxfController = new CXFServletControllerAdapter();
    		
    		return BAMServerProxy.wrap(cxfController, serviceDefinition);
    	} catch (Exception e) {
    		throw new RuntimeException(e);
    	}
    	
    }

    /**
     * @return the address where the service is (or will be) published
     */
    protected String getServiceAddress(ServiceDefinition serviceDefinition) {
        //Determine endpoint address to publish service on
        String serviceAddress = serviceDefinition.getServicePath();
        if (("/").equals(serviceAddress)){
        	serviceAddress = serviceAddress + serviceDefinition.getServiceName().getLocalPart();
        } else {
        	serviceAddress = serviceAddress + "/" + serviceDefinition.getServiceName().getLocalPart();
        }
        return serviceAddress;
    }

    /** 
     * This determines if the service has already been published on the CXF bus.
     * 
     * @return true if cxf server exists for this service.
     */
    protected boolean isServicePublished(String serviceAddress) {
    	
    	ServerRegistry serverRegistry = getCXFServerRegistry();
    	List<Server> servers = serverRegistry.getServers();
    	
    	for (Server server:servers){		
    		String endpointAddress = server.getEndpoint().getEndpointInfo().getAddress();
    		if (endpointAddress.contains(serviceAddress)){
    			LOG.info("Service already published on CXF, not republishing: " + serviceAddress);
    			return true;
    		}
    	}
    	
    	return false;
    }

    protected ServiceDefinition getServiceDefinition() {
    	return this.serviceDefinition;
    }

    protected Bus getCXFBus() {
    	return this.cxfBus;
    }

    protected ServerRegistry getCXFServerRegistry() {
    	return getCXFBus().getExtension(ServerRegistry.class);
    }

}
