/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ksb.messaging.web;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.action.InvalidCancelException;
import org.apache.struts.action.RequestProcessor;
import org.kuali.rice.coreservice.framework.CoreFrameworkServiceLocator;
import org.kuali.rice.coreservice.framework.parameter.ParameterConstants;
import org.kuali.rice.coreservice.framework.parameter.ParameterService;
import org.kuali.rice.krad.UserSession;
import org.kuali.rice.krad.util.CsrfValidator;
import org.kuali.rice.krad.util.GlobalVariables;
import org.kuali.rice.krad.util.KRADUtils;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;


/**
 * A RequestProcessor implementation for Struts which handles determining whether or not access
 * should be allowed to the requested KSB page.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class KSBStrutsRequestProcessor extends RequestProcessor {

	private static final Logger LOGGER = LogManager.getLogger(KSBStrutsRequestProcessor.class);

	private static final String KUALI_RICE_SYSTEM_NAMESPACE = "KR-SYS";

	@Override
	protected boolean processPreprocess(HttpServletRequest request,
			HttpServletResponse response) {
		final UserSession session = KRADUtils.getUserSessionFromRequest(request);

        if (session == null) {
            throw new IllegalStateException("the user session has not been established");
        }

        GlobalVariables.setUserSession(session);
        GlobalVariables.clear();
		return super.processPreprocess(request, response);
	}

	@Override
	protected boolean processValidate(HttpServletRequest request, HttpServletResponse response, ActionForm form, ActionMapping mapping) throws IOException, ServletException, InvalidCancelException {
		// need to make sure that we don't check CSRF until after the form is populated so that Struts will parse the
		// multipart parameters into the request if it's a multipart request
		final ParameterService parameterService = CoreFrameworkServiceLocator.getParameterService();
		if (parameterService.getParameterValueAsBoolean(KUALI_RICE_SYSTEM_NAMESPACE, ParameterConstants.ALL_COMPONENT, CsrfValidator.CSRF_PROTECTION_ENABLED_PARAM) && !CsrfValidator.validateCsrf(request, response)) {
			try {
				return false;
			} finally {
				// Special handling for multipart request
				if (form.getMultipartRequestHandler() != null) {
					if (LOGGER.isTraceEnabled()) {
						LOGGER.trace("  Rolling back multipart request");
					}

					form.getMultipartRequestHandler().rollback();
				}
			}
		}

		return super.processValidate(request, response, form, mapping);
	}

}
