/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.phs398ChecklistV11.PHS398ChecklistDocument;
import gov.grants.apply.forms.phs398ChecklistV11.PHS398ChecklistDocument.PHS398Checklist;
import gov.grants.apply.forms.phs398ChecklistV11.PHS398ChecklistDocument.PHS398Checklist.ApplicationType;
import gov.grants.apply.forms.phs398ChecklistV11.PHS398ChecklistDocument.PHS398Checklist.CertificationExplanation;
import gov.grants.apply.forms.phs398ChecklistV11.PHS398ChecklistDocument.PHS398Checklist.IncomeBudgetPeriod;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import gov.grants.apply.system.globalLibraryV20.HumanNameDataType;
import gov.grants.apply.system.globalLibraryV20.YesNoDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.rolodex.RolodexContract;
import org.kuali.coeus.common.api.rolodex.RolodexService;
import org.kuali.coeus.common.api.ynq.YnqConstant;
import org.kuali.coeus.common.budget.api.core.BudgetContract;
import org.kuali.coeus.common.budget.api.income.BudgetProjectIncomeContract;
import org.kuali.coeus.common.questionnaire.api.answer.AnswerHeaderContract;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.budget.ProposalDevelopmentBudgetExtContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * 
 * This class is used to generate XML Document object for grants.gov
 * PHS398ChecklistV1.1. This form is generated using XMLBean API's generated by
 * compiling PHS398ChecklistV1.1 schema.
 * 
 * @author Kuali Research Administration Team (kualidev@oncourse.iu.edu)
 */
@FormGenerator("PHS398ChecklistV1_1Generator")
public class PHS398ChecklistV1_1Generator extends PHS398ChecklistBaseGenerator<PHS398ChecklistDocument> {
	private static final int FEDERAL_ID_MAX_LENGTH = 30;

	List<? extends AnswerHeaderContract> answerHeaders;

    @Value("http://apply.grants.gov/forms/PHS398_Checklist-V1.1")
    private String namespace;

    @Value("PHS398_Checklist-V1.1")
    private String formName;

	@FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/PHS398_Checklist-V1.1.fo.xsl")
    private List<Resource> stylesheets;

    @Value("230")
    private int sortIndex;

    @Autowired
    @Qualifier("rolodexService")
    private RolodexService rolodexService;

	/**
	 * 
	 * This method returns PHS398ChecklistDocument object based on proposal
	 * development document which contains the PHS398ChecklistDocument
	 * informations
	 * ApplicationType,FederalID,ChangeOfPDPI,FormerPDName,ChangeOfInstitution,
	 * FormerInstitutionName,InventionsAndPatents, ProgramIncome and
	 * CertificationExplanation for a particular proposal
	 * 
	 * @return phsChecklistDocument {@link XmlObject} of type
	 *         PHS398ChecklistDocument.
	 */
	private PHS398ChecklistDocument getPHS398Checklist() {
		PHS398ChecklistDocument phsChecklistDocument = PHS398ChecklistDocument.Factory
				.newInstance();
		PHS398Checklist phsChecklist = PHS398Checklist.Factory.newInstance();
		phsChecklist.setFormVersion(FormVersion.v1_1.getVersion());
		ApplicationType.Enum applicationEnum = null;
		answerHeaders = getPropDevQuestionAnswerService().getQuestionnaireAnswerHeaders(pdDoc.getDevelopmentProposal().getProposalNumber());
		if (pdDoc.getDevelopmentProposal().getProposalType() != null
				&& Integer.parseInt(pdDoc.getDevelopmentProposal()
						.getProposalType().getCode()) < PROPOSAL_TYPE_CODE_6) {
			// Check <6 to ensure that if proposalType='TASk ORDER", it must not
			// set. THis is because enum ApplicationType has no
			// entry for TASK ORDER
			applicationEnum = ApplicationType.Enum.forInt(Integer
					.parseInt(pdDoc.getDevelopmentProposal()
							.getProposalType().getCode()));
		}
		phsChecklist.setApplicationType(applicationEnum);

		String federalId = getSubmissionInfoService().getFederalId(pdDoc.getDevelopmentProposal().getProposalNumber());
		if (federalId != null) {
			phsChecklist.setFederalID(StringUtils.substring(federalId.trim(), 0, FEDERAL_ID_MAX_LENGTH));
		}
		
		String pIChange = getAnswer(PROPOSAL_YNQ_QUESTION_114,answerHeaders);
        String pIChangeExplanation = getAnswer(PROPOSAL_YNQ_QUESTION_115,answerHeaders);
        if (YnqConstant.YES.code().equals(pIChange)) {
            phsChecklist.setIsChangeOfPDPI(YesNoDataType.Y_YES);
            if (pIChangeExplanation != null) {
                RolodexContract rolodex = rolodexService.getRolodex(Integer.valueOf(pIChangeExplanation));
                HumanNameDataType formerPDName = globLibV20Generator
                        .getHumanNameDataType(rolodex);
                if (formerPDName != null
                        && formerPDName.getFirstName() != null
                        && formerPDName.getLastName() != null) {
                    phsChecklist.setFormerPDName(formerPDName);
                }
            }
        } else {
            phsChecklist.setIsChangeOfPDPI(YesNoDataType.N_NO);
        }
        
		String institutionChange = getAnswer(PROPOSAL_YNQ_QUESTION_116,answerHeaders);
        String institutionChangeExplanation = getAnswer(PROPOSAL_YNQ_QUESTION_117,answerHeaders);
        
        if (YnqConstant.YES.code().equals(institutionChange)) {
            phsChecklist.setIsChangeOfInstitution(YesNoDataType.Y_YES);
            if (institutionChangeExplanation != null) {
                phsChecklist.setFormerInstitutionName(institutionChangeExplanation);
            }
        } else {
            phsChecklist.setIsChangeOfInstitution(YesNoDataType.N_NO);
        }
        
        String renewalApplication = getAnswer(PROPOSAL_YNQ_QUESTION_118,answerHeaders);
        boolean hasSubQuestionExplanation = false;
        if (renewalApplication != null && !renewalApplication.equals(NOT_ANSWERED)) {
            if (YnqConstant.YES.code().equals(renewalApplication)) {
                String inventionsConceived = getAnswer(PROPOSAL_YNQ_QUESTION_119,answerHeaders);
                if (YnqConstant.YES.code().equals(inventionsConceived)) {
                    phsChecklist.setIsInventionsAndPatents(YesNoDataType.Y_YES);
                    String reportedPreviously = getAnswer(PROPOSAL_YNQ_QUESTION_120,answerHeaders);
                    if (reportedPreviously != null && !reportedPreviously.equals(NOT_ANSWERED)) {
                        if (YnqConstant.YES.code().equals(reportedPreviously)) {
                            phsChecklist.setIsPreviouslyReported(YesNoDataType.Y_YES);
                        } else {
                            phsChecklist.setIsPreviouslyReported(YesNoDataType.N_NO);
                        }
                        hasSubQuestionExplanation = true;
                    }
                } else {
                    phsChecklist.setIsInventionsAndPatents(YesNoDataType.Y_YES);
                    if (hasSubQuestionExplanation) {
                        phsChecklist.setIsPreviouslyReported(YesNoDataType.N_NO);
                    }
                }
            } else {
                phsChecklist.setIsInventionsAndPatents(YesNoDataType.N_NO);
                if (hasSubQuestionExplanation) {
                    phsChecklist.setIsPreviouslyReported(YesNoDataType.N_NO);
                }
            }
        }

        ProposalDevelopmentBudgetExtContract budget = s2SCommonBudgetService.getBudget(pdDoc.getDevelopmentProposal());

		if (budget != null && budget.getBudgetProjectIncomes().size() > 0) {
			setProjectIncome(phsChecklist, budget);
		} else {
			phsChecklist.setProgramIncome(YesNoDataType.N_NO);
		}
		AttachedFileDataType attachedFileDataType = null;
		for (NarrativeContract narrative : pdDoc.getDevelopmentProposal()
				.getNarratives()) {
			if (narrative.getNarrativeType().getCode() != null
					&& Integer.parseInt(narrative.getNarrativeType().getCode()) == CERTIFICATIONS_ATTACHMENT_CODE) {
				attachedFileDataType = addAttachedFileType(narrative);
				if(attachedFileDataType != null){
					CertificationExplanation certExplanation = CertificationExplanation.Factory.newInstance();
					certExplanation.setCertifications(attachedFileDataType);
					phsChecklist.setCertificationExplanation(certExplanation);
				}
			}
		}
		phsChecklistDocument.setPHS398Checklist(phsChecklist);
		return phsChecklistDocument;
	}
	
	private void setProjectIncome(PHS398Checklist phsChecklist, BudgetContract budget) {
		Map<Integer, IncomeBudgetPeriod> incomeBudgetPeriodMap = new TreeMap<>();
		BigDecimal anticipatedAmount;
		for (BudgetProjectIncomeContract projectIncome : budget
				.getBudgetProjectIncomes()) {
			Integer budgetPeriodNumber = projectIncome.getBudgetPeriodNumber();
			IncomeBudgetPeriod incomeBudgPeriod = incomeBudgetPeriodMap
					.get(budgetPeriodNumber);
			if (incomeBudgPeriod == null) {
				incomeBudgPeriod = IncomeBudgetPeriod.Factory.newInstance();
				incomeBudgPeriod.setBudgetPeriod(budgetPeriodNumber);
				anticipatedAmount = BigDecimal.ZERO;
			} else {
				anticipatedAmount = incomeBudgPeriod.getAnticipatedAmount();
			}
			anticipatedAmount = anticipatedAmount.add(projectIncome.getProjectIncome().bigDecimalValue());
			incomeBudgPeriod.setAnticipatedAmount(anticipatedAmount);
			String description = getProjectIncomeDescription(projectIncome);
			if (description != null) {
				if (incomeBudgPeriod.getSource() != null) {
					incomeBudgPeriod.setSource(incomeBudgPeriod.getSource()
							+ ";" + description);
				} else {
					incomeBudgPeriod.setSource(description);
				}
			}
			incomeBudgetPeriodMap.put(budgetPeriodNumber, incomeBudgPeriod);
		}
		Collection<IncomeBudgetPeriod> incomeBudgetPeriodCollection = incomeBudgetPeriodMap
				.values();
		phsChecklist.setIncomeBudgetPeriodArray(incomeBudgetPeriodCollection
				.toArray(new IncomeBudgetPeriod[0]));
	}
	/*
	 * This method will get the project income description
	 */
	protected String getProjectIncomeDescription(BudgetProjectIncomeContract projectIncome) {
		String description = null;
		if (projectIncome.getDescription() != null) {
			if (projectIncome.getDescription().length() > PROJECT_INCOME_DESCRIPTION_MAX_LENGTH) {
				description = projectIncome.getDescription().substring(0,
						PROJECT_INCOME_DESCRIPTION_MAX_LENGTH);
			} else {
				description = projectIncome.getDescription();
			}
		}
		return description;
	}
	/**
	 * This method creates {@link XmlObject} of type
	 * {@link PHS398ChecklistDocument} by populating data from the given
	 * {@link ProposalDevelopmentDocumentContract}
	 * 
	 * @param proposalDevelopmentDocument
	 *            for which the {@link XmlObject} needs to be created
	 * @return {@link XmlObject} which is generated using the given
	 *         {@link ProposalDevelopmentDocumentContract}
	 */
	@Override
	public PHS398ChecklistDocument getFormObject(
			ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
		this.pdDoc = proposalDevelopmentDocument;
		return getPHS398Checklist();
	}

    public RolodexService getRolodexService() {
        return rolodexService;
    }

    public void setRolodexService(RolodexService rolodexService) {
        this.rolodexService = rolodexService;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

	@Override
	public List<Resource> getStylesheets() {
		return stylesheets;
	}

	public void setStylesheets(List<Resource> stylesheets) {
		this.stylesheets = stylesheets;
	}

	@Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

	@Override
	public DocumentFactory<PHS398ChecklistDocument> factory() {
		return PHS398ChecklistDocument.Factory;
	}
}
