/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */
package org.kuali.coeus.s2sgen.impl.generate.support.pdf.attachment;

import com.lowagie.text.pdf.*;
import java.io.IOException;
import java.util.*;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

/**
 * This tool extracts the internal PDF attachment key names.  These names are used in
 */
public class AttachmentTool {

    public static void main(String[] args) throws IOException {
        if (args.length < 1) {
            System.out.println("Usage: AttachmentTool /path/to/pdf/file.pdf");
            System.exit(1);
        }
        try (PdfReader reader = new PdfReader(args[0])) {
            final PdfDictionary catalog = reader.getCatalog();
            final PdfDictionary names = (PdfDictionary) PdfReader.getPdfObject(catalog.get(PdfName.NAMES));
            if (names != null) {
                final PdfDictionary embFiles = (PdfDictionary) PdfReader.getPdfObject(names.get(PdfName.EMBEDDEDFILES));
                if (embFiles != null) {
                    final Map<String, PdfObject> embMap = readTree(embFiles);
                    embMap.entrySet().stream()
                            .map(o -> entry(o.getKey(), (PdfDictionary) PdfReader.getPdfObject(o.getValue())))
                            .filter(e -> Objects.nonNull(e.getValue()))
                            .map(e -> entry(e.getKey(), getFilename(e.getValue())))
                            .filter(e -> Objects.nonNull(e.getKey()))
                            .sorted(Map.Entry.comparingByKey())
                            .forEach(e -> System.out.println(e.getKey() + " -> " + e.getValue()));
                }
            }
        }
    }

    private static String getFilename(PdfDictionary filespec) {
        final PdfString fn = (PdfString) PdfReader.getPdfObject(filespec.get(PdfName.F));
        if (fn == null) {
            return null;
        } else {
            return fn.toUnicodeString();
        }
    }

    private static void iterateItems(PdfDictionary dic, Map<String, PdfObject> items) {
        PdfArray nn = (PdfArray) PdfReader.getPdfObjectRelease(dic.get(PdfName.NAMES));
        if (nn != null) {
            for (int k = 0; k < nn.size(); ++k) {
                final PdfString s = (PdfString) PdfReader.getPdfObjectRelease(nn.getPdfObject(k++));
                items.put(s.toUnicodeString(), nn.getPdfObject(k));
            }
        } else if ((nn = (PdfArray) PdfReader.getPdfObjectRelease(dic.get(PdfName.KIDS))) != null) {
            for (int k = 0; k < nn.size(); ++k) {
                final PdfDictionary kid = (PdfDictionary) PdfReader.getPdfObjectRelease(nn.getPdfObject(k));
                iterateItems(kid, items);
            }
        }
    }

    /**
     * This method is Similar to {@link com.lowagie.text.pdf.PdfNameTree#readTree}
     * except that the PdfString is converted to a unicodeString
     */
    private static Map<String, PdfObject> readTree(PdfDictionary dic) {
        final Map<String, PdfObject> items = new HashMap<>();
        if (dic != null) {
            iterateItems(dic, items);
        }
        return items;
    }
}
