/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.print;

import com.lowagie.text.Font;
import com.lowagie.text.*;
import com.lowagie.text.pdf.*;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.fop.apps.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.coeus.propdev.api.s2s.S2SConfigurationService;
import org.kuali.coeus.s2sgen.api.core.ConfigurationConstants;
import org.kuali.coeus.s2sgen.api.core.S2SException;
import org.kuali.coeus.s2sgen.impl.util.SafeXmlUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;

import javax.xml.transform.*;
import javax.xml.transform.sax.SAXResult;
import javax.xml.transform.stream.StreamSource;
import java.awt.*;
import java.io.*;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.sql.Timestamp;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.*;
import java.util.stream.IntStream;

@Component("s2SPrintingService")
public class S2SPrintingServiceImpl implements S2SPrintingService {

    private static final Logger LOG = LogManager.getLogger(S2SPrintingServiceImpl.class);

    private static final char SPACE_SEPARATOR = 32;
    private static final int WHITESPACE_LENGTH_76 = 76;
    private static final int WHITESPACE_LENGTH_60 = 60;

    @Autowired
    @Qualifier("s2SConfigurationService")
    private S2SConfigurationService s2SConfigurationService;

    protected Map<String, byte[]> getPrintBytes(S2SPrintable printableArtifact) {

        String xml = printableArtifact.getXml();
        String name = printableArtifact.getName();
        try{
            String loggingEnable = s2SConfigurationService.getValueAsString(ConfigurationConstants.PRINT_LOGGING_ENABLE);
            if (Boolean.parseBoolean(loggingEnable)) {
                logPrintDetails(xml, name);
            }
        }catch(Exception ex){
            LOG.error(ex.getMessage());
        }

        Map<String, byte[]> pdfByteMap = new LinkedHashMap<>();
        final FopFactoryBuilder builder;
        try {
            builder = new FopFactoryBuilder(new URI(s2SConfigurationService.getValueAsString(ConfigurationConstants.APPLICATION_URL_KEY)));
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
        final FopFactory fopFactory = builder.build();

        // Apply all the style sheets to the xml document and generate the
        // PDF bytes
        if (CollectionUtils.isNotEmpty(printableArtifact.getXslTemplates())) {
            IntStream.range(0, printableArtifact.getXslTemplates().size()).forEach(i -> {
                final Resource xslt = printableArtifact.getXslTemplates().get(i);
                try {
                    if (xslt.getInputStream().available() <= 0) {
                        LOG.error("Stylesheet is not available");
                    } else {
                        createPdfWithFOP(xml, i + "_" + name, pdfByteMap, fopFactory, xslt);
                    }
                } catch (FOPException | TransformerException | IOException e) {
                    throw new S2SException(e.getMessage(), e);
                }
            });
        }

        // Add all the attachments.
        if (printableArtifact.getAttachments() != null) {
            pdfByteMap.putAll(printableArtifact.getAttachments());
        }
        return pdfByteMap;
    }

    protected void createPdfWithFOP(String xml, String name, Map<String, byte[]> pdfByteMap, FopFactory fopFactory, Resource xslt) throws FOPException,
            TransformerException, IOException {
        TransformerFactory factory = SafeXmlUtils.safeTransformerFactory();
        Transformer transformer = factory.newTransformer(new StreamSource(xslt.getInputStream()));
        FOUserAgent foUserAgent = fopFactory.newFOUserAgent();

        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
             ByteArrayInputStream inputStream = new ByteArrayInputStream(xml.getBytes(StandardCharsets.UTF_8))) {
            Source src = new StreamSource(inputStream);
            Fop fop = fopFactory.newFop(MimeConstants.MIME_PDF, foUserAgent, outputStream);
            Result res = new SAXResult(fop.getDefaultHandler());
            transformer.transform(src, res);
            byte[] pdfBytes = outputStream.toByteArray();
            if (pdfBytes.length > 0) {
                pdfByteMap.put(name, pdfBytes);
            }
        }

    }

    @Override
    public S2SFile print(S2SPrintable printableArtifacts) throws S2SException {
        List<S2SPrintable> printables = new ArrayList<>();
        printables.add(printableArtifacts);
        return print(printables);
    }

    @Override
    public S2SFile print(List<S2SPrintable> printableArtifactList) throws S2SException {
        S2SFile printablePdf;
        List<String> bookmarksList = new ArrayList<>();
        List<byte[]> pdfBaosList = new ArrayList<>();
        for (S2SPrintable printableArtifact : printableArtifactList) {
            Map<String, byte[]> printBytes = getPrintBytes(printableArtifact);
            for (String bookmark : printBytes.keySet()) {
                byte[] pdfBytes = printBytes.get(bookmark);
                if (isPdfGoodToMerge(pdfBytes)) {
                    bookmarksList.add(bookmark);
                    pdfBaosList.add(pdfBytes);
                }
            }
        }

        printablePdf = new S2SFile();
        byte[] mergedPdfBytes = mergePdfBytes(pdfBaosList, bookmarksList, false);

        // If there is a stylesheet issue, the pdf bytes will be null. To avoid an exception
        // initialize to an empty array before sending the content back
        if (mergedPdfBytes == null) {
            mergedPdfBytes = new byte[0];
        }

        printablePdf.setData(mergedPdfBytes);
        StringBuilder fileName = new StringBuilder();
        fileName.append(getReportName());
        fileName.append(PrintConstants.PDF_FILE_EXTENSION);
        printablePdf.setName(fileName.toString());
        printablePdf.setType(PrintConstants.PDF_REPORT_CONTENT_TYPE);

        final boolean loggingEnable = s2SConfigurationService.getValueAsBoolean(ConfigurationConstants.PRINT_PDF_LOGGING_ENABLE);
        if (loggingEnable) {
            logPdfPrintDetails(printablePdf);
        }

        return printablePdf;
    }

    protected boolean isPdfGoodToMerge(byte[] pdfBytes) throws S2SException {
        try (PdfReader reader = new PdfReader(pdfBytes)) {
            return true;
        }
        catch (IOException e) {
            throw new S2SException(e);
        }
    }

    protected String getReportName() {
        String dateString = new Date().toString();
        return StringUtils.deleteWhitespace(dateString);
    }

    @Override
    public byte[] mergePdfBytes(List<byte[]> pdfBytesList, List<String> bookmarksList, boolean headerFooterRequired) throws S2SException {
        Document document = null;
        PdfWriter writer = null;
        ByteArrayOutputStream mergedPdfReport = new ByteArrayOutputStream();
        int totalNumOfPages = 0;
        PdfReader[] pdfReaderArr = new PdfReader[pdfBytesList.size()];
        int pdfReaderCount = 0;
        for (byte[] fileBytes : pdfBytesList) {
            LOG.debug("File Size " + fileBytes.length + " For " + bookmarksList.get(pdfReaderCount));
            PdfReader reader;
            try {
                reader = new PdfReader(fileBytes);
                pdfReaderArr[pdfReaderCount] = reader;
                pdfReaderCount = pdfReaderCount + 1;
                totalNumOfPages += reader.getNumberOfPages();
            }
            catch (IOException e) {
                LOG.error(e.getMessage(), e);
            }
        }
        HeaderFooter footer = null;
        if (headerFooterRequired) {
            Calendar calendar = Calendar.getInstance();
            String dateString = formateCalendar(calendar);
            StringBuilder footerPhStr = new StringBuilder();
            footerPhStr.append(" of ");
            footerPhStr.append(totalNumOfPages);
            footerPhStr.append(getWhitespaceString(WHITESPACE_LENGTH_76));
            footerPhStr.append(getWhitespaceString(WHITESPACE_LENGTH_76));
            footerPhStr.append(getWhitespaceString(WHITESPACE_LENGTH_60));
            footerPhStr.append(dateString);
            Font font = FontFactory.getFont(FontFactory.TIMES, 8, Font.NORMAL, Color.BLACK);
            Phrase beforePhrase = new Phrase("Page ", font);
            Phrase afterPhrase = new Phrase(footerPhStr.toString(), font);
            footer = new HeaderFooter(beforePhrase, afterPhrase);
            footer.setAlignment(Element.ALIGN_BASELINE);
            footer.setBorderWidth(0f);
        }
        for (int count = 0; count < pdfReaderArr.length; count++) {
            PdfReader reader = pdfReaderArr[count];
            int nop;
            if (reader == null) {
                LOG.debug("Empty PDF byetes found for " + bookmarksList.get(count));
                continue;
            }
            else {
                nop = reader.getNumberOfPages();
            }

            if (count == 0) {
                document = nop > 0 ? new Document(reader.getPageSizeWithRotation(1))
                        : new Document();
                try {
                    writer = PdfWriter.getInstance(document, mergedPdfReport);
                }
                catch (DocumentException e) {
                    LOG.error(e.getMessage(), e);
                    throw new S2SException(e.getMessage(), e);
                }
                if (footer != null) {
                    document.setFooter(footer);
                }
                document.open();
            }

            PdfContentByte cb = writer.getDirectContent();
            int pageCount = 0;
            while (pageCount < nop) {
                document.setPageSize(reader.getPageSize(++pageCount));
                document.newPage();
                if (footer != null) {
                    document.setFooter(footer);
                }
                PdfImportedPage page = writer.getImportedPage(reader, pageCount);

                cb.addTemplate(page, 1, 0, 0, 1, 0, 0);


                PdfOutline root = cb.getRootOutline();
                if (pageCount == 1) {
                    String pageName = bookmarksList.get(count);
                    cb.addOutline(new PdfOutline(root, new PdfDestination(PdfDestination.FITH), pageName), pageName);
                }
            }
        }
        if (document != null) {
            try {
                document.close();
                return mergedPdfReport.toByteArray();
            }
            catch (Exception e) {
                LOG.error("Exception occured because the generated PDF document has no pages", e);
            }
        }
        return null;
    }


    protected String formateCalendar(Calendar calendar) {
        DateFormat dateFormat = new SimpleDateFormat("M/d/yy h:mm a");
        return dateFormat.format(calendar.getTime());
    }

    protected String getWhitespaceString(int length) {
        StringBuilder sb = new StringBuilder();
        char[] whiteSpace = new char[length];
        Arrays.fill(whiteSpace, SPACE_SEPARATOR);
        sb.append(whiteSpace);
        return sb.toString();
    }

    protected void logPrintDetails(String xmlString, String key) {
        String loggingDirectory = s2SConfigurationService.getValueAsString(ConfigurationConstants.PRINT_LOGGING_DIRECTORY);
        if (loggingDirectory != null) {
            String dateString = new Timestamp(new Date().getTime()).toString();
            String reportName = StringUtils.deleteWhitespace(key);
            String createdTime = StringUtils.replaceChars(StringUtils.deleteWhitespace(dateString), PrintConstants.COLON, PrintConstants.UNDERSCORE);
            File dir = getLoggingDir(loggingDirectory);
            File file = new File(dir , reportName + PrintConstants.UNDERSCORE + createdTime + PrintConstants.XML_FILE_EXTENSION);
            try(BufferedWriter out = new BufferedWriter(new FileWriter(file))) {
                out.write(xmlString);
            } catch (IOException e) {
                    LOG.error(e.getMessage(), e);
            }
        }
    }

    protected void logPdfPrintDetails(S2SFile pdf) {
        final String loggingDirectory = s2SConfigurationService.getValueAsString(ConfigurationConstants.PRINT_LOGGING_DIRECTORY);
        if (loggingDirectory != null && pdf != null && pdf.getData() != null) {
            final String dateString = new Timestamp(new Date().getTime()).toString();
            final String createdTime = StringUtils.replaceChars(StringUtils.deleteWhitespace(dateString), PrintConstants.COLON, PrintConstants.UNDERSCORE);
            final String fileName = StringUtils.replaceChars(StringUtils.deleteWhitespace(pdf.getName().replace(PrintConstants.PDF_FILE_EXTENSION, "")), PrintConstants.COLON, PrintConstants.UNDERSCORE) + PrintConstants.UNDERSCORE +
                    createdTime + PrintConstants.PDF_FILE_EXTENSION;
            try(FileOutputStream fos = new FileOutputStream(new File(getLoggingDir(loggingDirectory), fileName))) {
                fos.write(pdf.getData());
                fos.flush();
            } catch (IOException e) {
                LOG.error(e.getMessage(), e);
            }
        }
    }

    protected File getLoggingDir(String loggingDirectory) {
        File dir = new File(loggingDirectory);
        if(!dir.exists() || !dir.isDirectory()){
            dir.mkdirs();
        }
        return dir;
    }

    public S2SConfigurationService getS2SConfigurationService() {
        return s2SConfigurationService;
    }

    public void setS2SConfigurationService(S2SConfigurationService s2SConfigurationService) {
        this.s2SConfigurationService = s2SConfigurationService;
    }
}
