/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.coeus.personProfile.PersonProfileListDocument;
import gov.grants.apply.forms.nsfKeyPersonExpanded20V20.NSFKeyPersonExpanded20Document;
import gov.grants.apply.forms.nsfKeyPersonExpanded20V20.NSFKeyPersonExpanded20Document.NSFKeyPersonExpanded20;
import gov.grants.apply.forms.nsfKeyPersonExpanded20V20.NSFKeyPersonExpanded20Document.NSFKeyPersonExpanded20.AdditionalProfilesAttached;
import gov.grants.apply.forms.nsfKeyPersonExpanded20V20.PersonProfileDataType;
import gov.grants.apply.forms.nsfKeyPersonExpanded20V20.PersonProfileDataType.Profile;
import gov.grants.apply.forms.nsfKeyPersonExpanded20V20.PersonProfileDataType.Profile.OtherProjectRoleCategory;
import gov.grants.apply.forms.nsfKeyPersonExpanded20V20.ProjectRoleDataType;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.rolodex.RolodexContract;
import org.kuali.coeus.common.api.rolodex.RolodexService;
import org.kuali.coeus.common.api.unit.UnitContract;
import org.kuali.coeus.common.api.unit.UnitRepositoryService;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.core.DevelopmentProposalContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.person.attachment.ProposalPersonBiographyContract;
import org.kuali.coeus.s2sgen.api.core.S2SException;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.*;
import org.kuali.coeus.s2sgen.impl.person.S2SProposalPersonService;
import org.kuali.coeus.s2sgen.impl.person.S2sDivisionService;
import org.kuali.coeus.s2sgen.impl.print.GenericPrintable;
import org.kuali.coeus.s2sgen.impl.print.S2SPrintingService;
import org.kuali.coeus.s2sgen.impl.util.FieldValueConstants;
import org.kuali.coeus.sys.api.model.KcFile;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

@FormGenerator("NSFKeyPersonExpandedV2_0Generator")
public class NSFKeyPersonExpandedV2_0Generator extends S2SBaseFormGenerator<NSFKeyPersonExpanded20Document> implements S2SFormGeneratorPdfFillable<NSFKeyPersonExpanded20Document> {

    private static final Logger LOG = LogManager.getLogger(NSFKeyPersonExpandedV2_0Generator.class);

    protected static final int TITLE_MAX_LENGTH = 45;
    protected static final int ROLE_DESCRIPTION_MAX_LENGTH = 40;
    private static final int MAX_KEY_PERSON_COUNT = 100;
    protected static final int BIOSKETCH_DOC_TYPE = 16;
    protected static final int CURRENTPENDING_DOC_TYPE = 17;
    protected static final int COLLABORATOR_DOC_TYPE = 148;
    protected static final String BIOSKETCH_TYPE = "1";
    protected static final String COLLABORATOR_TYPE = "6";
    protected static final String CURRENT_PENDING_TYPE = "2";
    private static final String COMMENT = "Auto generated document for ";
    private static final String BIOSKETCH_COMMENT = "BIOSKETCH";
    private static final String CURRENT_PENDING_COMMENT = "CURRENTPENDING";
    private static final String COLLABORATOR_COMMENT = "COLLABORATOR";
    protected static final String PROFILE_COMMENT = "PROFILE";
    protected static final int PROFILE_TYPE = 18;
    protected static final String NIH_CO_INVESTIGATOR = "Co-Investigator";
    protected static final int DEPARTMENT_NAME_MAX_LENGTH = 30;
    private static final int ADDRESS_LINE_MAX_LENGTH = 55;

    protected RolodexContract rolodex;
    protected List<ProposalPersonContract> extraPersons = null;
    protected String pIPersonOrRolodexId = null;

    @Autowired
    @Qualifier("s2SPrintingService")
    private S2SPrintingService s2SPrintingService;

    @Autowired
    @Qualifier("s2SProposalPersonService")
    protected S2SProposalPersonService s2SProposalPersonService;

    @Autowired
    @Qualifier("s2sDivisionService")
    private S2sDivisionService s2sDivisionService;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/additionalkeypersonprofiles.xsl")
    private Resource additionalkeypersonprofilesStyleSheet;

    @Value("http://apply.grants.gov/forms/NSF_KeyPersonExpanded_2_0-V2.0")
    private String namespace;

    @Value("NSF_KeyPersonExpanded_2_0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/NSF_KeyPersonExpanded-V2.0.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/NSF_KeyPersonExpanded_2_0-V2.0.pdf")
    private Resource pdfForm;

    @Value("155")
    private int sortIndex;

    @Autowired
    @Qualifier("rolodexService")
    private RolodexService rolodexService;

    @Autowired
    @Qualifier("unitRepositoryService")
    private UnitRepositoryService unitRepositoryService;

    protected void saveKeyPersonAttachmentsToProposal() {
        if (!CollectionUtils.isEmpty(extraPersons)) {
            saveKeyPersonAttachments();
            saveKeypersonProfileObject();
        }
    }

    private void saveKeyPersonAttachments() {
        List<String> bioSketchBookMarks = new ArrayList<>();
        List<String> curPendBookMarks = new ArrayList<>();
        List<String> collaboratorBookMarks = new ArrayList<>();
        List<byte[]> bioSketchDataList = new ArrayList<>();
        List<byte[]> curPendDataList = new ArrayList<>();
        List<byte[]> collaboratorDataList = new ArrayList<>();

        for (ProposalPersonContract proposalPerson : extraPersons) {
            setBookMarkAndData(bioSketchBookMarks, bioSketchDataList, proposalPerson, BIOSKETCH_TYPE);
            setBookMarkAndData(curPendBookMarks, curPendDataList, proposalPerson, CURRENT_PENDING_TYPE);
            setBookMarkAndData(collaboratorBookMarks, collaboratorDataList, proposalPerson, COLLABORATOR_TYPE);
        }
        byte[] bioSketchData;
        byte[] curPendData;
        byte[] collaboratorData;
        try {
            bioSketchData = s2SPrintingService.mergePdfBytes(bioSketchDataList, bioSketchBookMarks, true);
            curPendData = s2SPrintingService.mergePdfBytes(curPendDataList, curPendBookMarks, true);
            collaboratorData = s2SPrintingService.mergePdfBytes(collaboratorDataList, collaboratorBookMarks, true);
            String fileName;
            if (bioSketchData != null && bioSketchData.length > 0) {
                fileName = pdDoc.getDevelopmentProposal().getProposalNumber() + "_" + BIOSKETCH_COMMENT + ".pdf";
                saveNarrative(bioSketchData, "" + BIOSKETCH_DOC_TYPE, fileName, COMMENT + BIOSKETCH_COMMENT);
            }
            if (curPendData != null && curPendData.length > 0) {
                fileName = pdDoc.getDevelopmentProposal().getProposalNumber() + "_" + CURRENT_PENDING_COMMENT + ".pdf";
                saveNarrative(curPendData, "" + CURRENTPENDING_DOC_TYPE, fileName, COMMENT + CURRENT_PENDING_COMMENT);
            }
            if (collaboratorData != null && collaboratorData.length > 0) {
                fileName = pdDoc.getDevelopmentProposal().getProposalNumber() + "_" + COLLABORATOR_COMMENT + ".pdf";
                saveNarrative(collaboratorData, "" + COLLABORATOR_DOC_TYPE, fileName, COMMENT + COLLABORATOR_COMMENT);
            }
        } catch (S2SException e) {
            LOG.error("Auto generation of Biosketch/Currend Pending report for extra Keypersons is failed", e);
        }
    }

    private void setBookMarkAndData(List<String> bookMarksList,
                                    List<byte[]> dataList, ProposalPersonContract proposalPerson, String docType) {
        final String personId;
        if (proposalPerson.getPersonId() != null && proposalPerson.getPersonId().length() > 0) {
            personId = proposalPerson.getPersonId();
        } else {
            personId = "" + proposalPerson.getRolodexId();
        }

        for (ProposalPersonBiographyContract personBiography : getPernonnelAttachments(pdDoc, proposalPerson, docType)) {
            byte[] content = personBiography.getPersonnelAttachment().getData();
            if (content != null && content.length > 0) {
                dataList.add(content);
                bookMarksList.add(personId);
            }
        }
    }

    private List<ProposalPersonBiographyContract> getPernonnelAttachments(ProposalDevelopmentDocumentContract pdDoc, ProposalPersonContract proposalPerson, String documentType) {
        List<ProposalPersonBiographyContract> result = new ArrayList<>();
        for (ProposalPersonBiographyContract proposalPersonBiography : pdDoc.getDevelopmentProposal().getPropPersonBios()) {
            String personId = proposalPerson.getPersonId();
            Integer rolodexId = proposalPerson.getRolodexId();
            if (personId != null && proposalPersonBiography.getPersonId() != null && proposalPersonBiography.getPersonId().equals(personId) && documentType.equals(proposalPersonBiography.getPropPerDocType().getCode())) {
                result.add(proposalPersonBiography);
            } else if (rolodexId != null && proposalPersonBiography.getRolodexId() != null
                    && proposalPersonBiography.getRolodexId().toString().equals(rolodexId.toString())
                    && documentType.equals(proposalPersonBiography.getPropPerDocType().getCode())) {
                result.add(proposalPersonBiography);
            }
        }
        return result;
    }

    protected PersonProfileListDocument.PersonProfileList.ExtraKeyPerson[] getExtraKeyPersons() {
        List<PersonProfileListDocument.PersonProfileList.ExtraKeyPerson> extraPersonList = new ArrayList<>();

        for (ProposalPersonContract proposalPerson : extraPersons) {

            PersonProfileListDocument.PersonProfileList.ExtraKeyPerson extraPerson = PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.Factory.newInstance();

            extraPerson.setName(getExtraPersonName(proposalPerson));
            extraPerson.setAddress(getExtraPersonAddress(proposalPerson));
            if (proposalPerson.getPrimaryTitle() != null && proposalPerson.getPrimaryTitle().length() > TITLE_MAX_LENGTH)
                extraPerson.setTitle(proposalPerson.getPrimaryTitle().substring(0, TITLE_MAX_LENGTH));
            else {
                extraPerson.setTitle(proposalPerson.getPrimaryTitle());
            }

            if (proposalPerson.getProposalPersonRoleId() != null) {
                if (proposalPerson.isPrincipalInvestigator()) {
                    extraPerson.setProjectRole(PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.ProjectRole.PD_PI);
                } else if (proposalPerson.isCoInvestigator()) {
                    if (isSponsorNIH(pdDoc)) {
                        extraPerson.setProjectRole(PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.ProjectRole.OTHER_SPECIFY);
                        extraPerson.setOtherProjectRoleCategory(NIH_CO_INVESTIGATOR);
                    } else {
                        extraPerson.setProjectRole(PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.ProjectRole.CO_PD_PI);
                    }
                } else {
                    final String otherRole;
                    if (proposalPerson.getProjectRole() != null && proposalPerson.getProjectRole().length() > ROLE_DESCRIPTION_MAX_LENGTH) {
                        otherRole = proposalPerson.getProjectRole().substring(0, ROLE_DESCRIPTION_MAX_LENGTH);
                    } else {
                        otherRole = proposalPerson.getProjectRole();
                    }
                    extraPerson.setOtherProjectRoleCategory(otherRole);
                    extraPerson.setProjectRole(PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.ProjectRole.OTHER_SPECIFY);
                }
            }

            if (proposalPerson.getEraCommonsUserName() != null) {
                extraPerson.setCredential(proposalPerson.getEraCommonsUserName());
            }
            setDepartmentName(extraPerson);
            setDivisionName(extraPerson);
            if (proposalPerson.getEmailAddress() != null) {
                extraPerson.setEmail(proposalPerson.getEmailAddress());
            }
            if (StringUtils.isNotEmpty(proposalPerson.getFaxNumber())) {
                extraPerson.setFax(proposalPerson.getFaxNumber());
            }
            UnitContract unit = unitRepositoryService.findUnitByUnitNumber(proposalPerson.getHomeUnit());

            if (unit != null && unit.getUnitName() != null) {
                extraPerson.setOrganizationName(unit.getUnitName());
            }
            if (proposalPerson.getOfficePhone() != null) {
                extraPerson.setPhone(proposalPerson.getOfficePhone());
            }
            AttachedFileDataType bioSketchAttachment = getPersonnelAttachments(pdDoc, proposalPerson.getPersonId(), proposalPerson.getRolodexId(), BIOSKETCH_TYPE);
            if (bioSketchAttachment != null) {
                extraPerson.setBioSketchAttached(PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.BioSketchAttached.YES);
            }

            AttachedFileDataType curPendingAttachment = getPersonnelAttachments(pdDoc, proposalPerson.getPersonId(), proposalPerson.getRolodexId(), CURRENT_PENDING_TYPE);
            if (curPendingAttachment != null) {
                extraPerson.setSupportsAttached(PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.SupportsAttached.YES);
            }

            AttachedFileDataType collaboratorAttachment = getPersonnelAttachments(pdDoc, proposalPerson.getPersonId(), proposalPerson.getRolodexId(), COLLABORATOR_TYPE);
            if (collaboratorAttachment != null) {
                extraPerson.setCollaboratorAttached(PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.CollaboratorAttached.YES);
            }

            extraPersonList.add(extraPerson);
        }
        return extraPersonList.toArray(new PersonProfileListDocument.PersonProfileList.ExtraKeyPerson[0]);
    }

    private PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.Address getExtraPersonAddress(
            ProposalPersonContract proposalPerson) {
        PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.Address address = PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.Address.Factory.newInstance();
        if (proposalPerson.getAddressLine1() != null) {
            if (proposalPerson.getAddressLine1().length() > ADDRESS_LINE_MAX_LENGTH) {
                address.setStreet1(proposalPerson.getAddressLine1().substring(0, ADDRESS_LINE_MAX_LENGTH));
            } else {
                address.setStreet1(proposalPerson.getAddressLine1());
            }
        }
        if (proposalPerson.getAddressLine2() != null) {
            if (proposalPerson.getAddressLine2().length() > ADDRESS_LINE_MAX_LENGTH) {
                address.setStreet2(proposalPerson.getAddressLine2().substring(0, ADDRESS_LINE_MAX_LENGTH));
            } else {
                address.setStreet2(proposalPerson.getAddressLine2());
            }
        }
        if (proposalPerson.getCity() != null) {
            address.setCity(proposalPerson.getCity());
        }
        if (proposalPerson.getCounty() != null) {
            address.setCounty(proposalPerson.getCounty());
        }

        if (proposalPerson.getPostalCode() != null) {
            address.setZipCode(proposalPerson.getPostalCode());
        }

        if (proposalPerson.getCountryCode() != null) {
            PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.Address.Country.Enum county = PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.Address.Country.Enum.forString(proposalPerson.getCountryCode());
            address.setCountry(county);
        }
        if (proposalPerson.getState() != null) {
            address.setState(proposalPerson.getState());
        }
        return address;
    }

    private PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.Name getExtraPersonName(ProposalPersonContract proposalPerson) {
        PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.Name name = PersonProfileListDocument.PersonProfileList.ExtraKeyPerson.Name.Factory.newInstance();
        if (proposalPerson.getFirstName() != null) {
            name.setFirstName(proposalPerson.getFirstName());
        }
        if (proposalPerson.getMiddleName() != null) {
            name.setMiddleName(proposalPerson.getMiddleName());
        }
        if (proposalPerson.getLastName() != null) {
            name.setLastName(proposalPerson.getLastName());
        }
        return name;
    }

    private void setDivisionName(PersonProfileListDocument.PersonProfileList.ExtraKeyPerson extraPerson) {
        extraPerson.setDivisionName("");
    }

    private void setDepartmentName(PersonProfileListDocument.PersonProfileList.ExtraKeyPerson extraPerson) {
        extraPerson.setDepartmentName("");
    }

    private void saveKeypersonProfileObject() {

        if (!CollectionUtils.isEmpty(extraPersons)) {
            PersonProfileListDocument.PersonProfileList extraPersonProfileList = PersonProfileListDocument.PersonProfileList.Factory.newInstance();

            extraPersonProfileList.setProposalNumber(pdDoc.getDevelopmentProposal().getProposalNumber());
            extraPersonProfileList.setExtraKeyPersonArray(getExtraKeyPersons());

            PersonProfileListDocument extraPersonDoc = PersonProfileListDocument.Factory.newInstance();
            extraPersonDoc.setPersonProfileList(extraPersonProfileList);
            String xmlData = extraPersonDoc.xmlText();

            GenericPrintable printable = new GenericPrintable();
            printable.setXslTemplates(Collections.singletonList(additionalkeypersonprofilesStyleSheet));
            printable.setName("");
            printable.setXml(xmlData);
            try {
                KcFile printData = s2SPrintingService.print(printable);
                String fileName = pdDoc.getDevelopmentProposal().getProposalNumber() + "_" + PROFILE_COMMENT + ".pdf";
                saveNarrative(printData.getData(), "" + PROFILE_TYPE, fileName, COMMENT + PROFILE_COMMENT);
            } catch (S2SException e) {
                LOG.error("Auto generation of Profile attachment for extra Keypersons failed", e);
            }
        }
    }


    private NSFKeyPersonExpanded20Document getNSFKeyPersonExpanded() {
        NSFKeyPersonExpanded20Document nsfKeyPersonExpandedDocument = NSFKeyPersonExpanded20Document.Factory.newInstance();
        NSFKeyPersonExpanded20 nsfKeyPersonExpanded = NSFKeyPersonExpanded20.Factory.newInstance();
        setNSFKeyPersonExpandedAttributes(nsfKeyPersonExpanded);
        nsfKeyPersonExpandedDocument.setNSFKeyPersonExpanded20(nsfKeyPersonExpanded);

        return nsfKeyPersonExpandedDocument;
    }

    private void setNSFKeyPersonExpandedAttributes(NSFKeyPersonExpanded20 nsfKeyPersonExpanded) {
        nsfKeyPersonExpanded.setFormVersion(FormVersion.v2_0.getVersion());
        nsfKeyPersonExpanded.setPDPI(getPersonProfilePI());
        PersonProfileDataType[] keyPersonArray = getPersonProfileKeyPerson();
        if (keyPersonArray.length > 0) {
            nsfKeyPersonExpanded.setKeyPersonArray(keyPersonArray);
        }
        saveKeyPersonAttachmentsToProposal();
        if (extraPersons.size() > 0) {
            for (ProposalPersonContract extraPerson : extraPersons) {
                setBioSketchAttchment(nsfKeyPersonExpanded, extraPerson);
                setCurrentPendingTypeAttachment(nsfKeyPersonExpanded, extraPerson);
            }
            for (NarrativeContract narrative : pdDoc.getDevelopmentProposal().getNarratives()) {
                if (narrative.getNarrativeType().getCode() != null) {
                    if (Integer.parseInt(narrative.getNarrativeType().getCode()) == PROFILE_TYPE) {
                        setProfileTypeAttachment(nsfKeyPersonExpanded, narrative);
                    }
                }
            }
        }
    }


    private void setProfileTypeAttachment(NSFKeyPersonExpanded20 nsfKeyPersonExpanded, NarrativeContract narrative) {
        AttachedFileDataType attachedFileDataType = addAttachedFileType(narrative);
        if (attachedFileDataType != null) {
            AdditionalProfilesAttached additionalProfilesAttached = AdditionalProfilesAttached.Factory.newInstance();
            additionalProfilesAttached.setAdditionalProfileAttached(attachedFileDataType);
            nsfKeyPersonExpanded.setAdditionalProfilesAttached(additionalProfilesAttached);
        }
    }

    private void setCurrentPendingTypeAttachment(NSFKeyPersonExpanded20 nsfKeyPersonExpanded, ProposalPersonContract extraPerson) {
        AttachedFileDataType supportAttachment = getPersonnelAttachments(pdDoc, extraPerson.getPersonId(), extraPerson.getRolodexId(), CURRENT_PENDING_TYPE);
        if (supportAttachment != null) {
            NSFKeyPersonExpanded20.SupportsAttached supportsAttached = NSFKeyPersonExpanded20.SupportsAttached.Factory.newInstance();
            supportsAttached.setSupportAttached(supportAttachment);
            nsfKeyPersonExpanded.setSupportsAttached(supportsAttached);
        }
    }

    private void setBioSketchAttchment(NSFKeyPersonExpanded20 nsfKeyPersonExpanded, ProposalPersonContract extraPerson) {
        NSFKeyPersonExpanded20.BioSketchsAttached personBioSketch = NSFKeyPersonExpanded20.BioSketchsAttached.Factory.newInstance();
        AttachedFileDataType bioSketchAttachment = getPersonnelAttachments(pdDoc, extraPerson.getPersonId(), extraPerson.getRolodexId(), BIOSKETCH_TYPE);
        personBioSketch.setBioSketchAttached(bioSketchAttachment);
        nsfKeyPersonExpanded.setBioSketchsAttached(personBioSketch);
    }


    private PersonProfileDataType getPersonProfilePI() {
        PersonProfileDataType profileDataType = PersonProfileDataType.Factory.newInstance();
        profileDataType.setProfile(Profile.Factory.newInstance());
        ProposalPersonContract PI = s2SProposalPersonService.getPrincipalInvestigator(pdDoc);
        if (PI != null) {
            setPersonalProfileDetailsToProfile(profileDataType, PI);
        }
        return profileDataType;
    }

    private void setPersonalProfileDetailsToProfile(PersonProfileDataType profileDataType, ProposalPersonContract principalInvestigator) {
        Profile profile = profileDataType.getProfile();
        assignRolodexId(principalInvestigator);
        profile.setName(globLibV20Generator.getHumanNameDataType(principalInvestigator));
        setDirectoryTitleToProfile(profile, principalInvestigator);
        profile.setAddress(globLibV20Generator.getAddressDataTypeV3(principalInvestigator));
        profile.setPhone(principalInvestigator.getOfficePhone());
        if (StringUtils.isNotEmpty(principalInvestigator.getFaxNumber())) {
            profile.setFax(principalInvestigator.getFaxNumber());
        }
        profile.setEmail(principalInvestigator.getEmailAddress());
        DevelopmentProposalContract developmentProposal = pdDoc.getDevelopmentProposal();
        setOrganizationName(profile, developmentProposal, principalInvestigator);
        setDepartmentNameToProfile(profile, principalInvestigator);
        final String divisionName = getS2sDivisionService().getDivision(principalInvestigator);
        if (divisionName != null) {
            profile.setDivisionName(divisionName);
        }
        profile.setNSFID(principalInvestigator.getNsfId());
        profile.setProjectRole(ProjectRoleDataType.PD_PI);
        setAttachments(profile, principalInvestigator);
    }

    private void setDepartmentNameToProfile(Profile profile, ProposalPersonContract PI) {
        if (PI.getHomeUnit() != null && PI.getPerson() != null && PI.getPerson().getUnit() != null) {
            final String departmentName = PI.getPerson().getUnit().getUnitName();
            profile.setDepartmentName(StringUtils.substring(departmentName, 0, DEPARTMENT_NAME_MAX_LENGTH));
        } else {
            DevelopmentProposalContract developmentProposal = pdDoc.getDevelopmentProposal();
            profile.setDepartmentName(StringUtils.substring(developmentProposal.getOwnedByUnit().getUnitName(), 0, DEPARTMENT_NAME_MAX_LENGTH));
        }
    }

    private void setDirectoryTitleToProfile(Profile profile, ProposalPersonContract PI) {
        if (PI.getDirectoryTitle() != null) {
            if (PI.getDirectoryTitle().length() > TITLE_MAX_LENGTH) {
                profile.setTitle(PI.getDirectoryTitle().substring(0, TITLE_MAX_LENGTH));
            } else {
                profile.setTitle(PI.getDirectoryTitle());
            }
        }
    }

    private void assignRolodexId(ProposalPersonContract PI) {
        if (PI.getPersonId() != null) {
            pIPersonOrRolodexId = PI.getPersonId();
            rolodex = null;
        } else if (PI.getRolodexId() != null) {
            pIPersonOrRolodexId = PI.getRolodexId().toString();
            rolodex = rolodexService.getRolodex(Integer.valueOf(pIPersonOrRolodexId));
        }
    }

    private void setAttachments(Profile profile, ProposalPersonContract PI) {
        setBioSketchAttachment(profile, PI);
        setCurrentPendingAttachment(profile, PI);
        setCollaboratorsAttachment(profile, PI);
    }

    private void setCurrentPendingAttachment(Profile profile, ProposalPersonContract PI) {
        profile.setSupportsAttached(Profile.SupportsAttached.Factory.newInstance());
        AttachedFileDataType supportAttachment = getPersonnelAttachments(pdDoc, PI.getPersonId(), PI.getRolodexId(), CURRENT_PENDING_TYPE);
        if (supportAttachment == null) {
            supportAttachment = AttachedFileDataType.Factory.newInstance();
        }
        profile.getSupportsAttached().setSupportAttached(supportAttachment);
    }

    private void setBioSketchAttachment(Profile profile, ProposalPersonContract PI) {
        profile.setBioSketchsAttached(Profile.BioSketchsAttached.Factory.newInstance());
        AttachedFileDataType bioSketchAttachment = getPersonnelAttachments(pdDoc, PI.getPersonId(), PI.getRolodexId(), BIOSKETCH_TYPE);
        if (bioSketchAttachment == null) {
            bioSketchAttachment = AttachedFileDataType.Factory.newInstance();
        }
        profile.getBioSketchsAttached().setBioSketchAttached(bioSketchAttachment);
    }

    private void setCollaboratorsAttachment(Profile profile, ProposalPersonContract PI) {
        profile.setCollaboratorsAttached(Profile.CollaboratorsAttached.Factory.newInstance());
        AttachedFileDataType collaboratorAttachment = getPersonnelAttachments(pdDoc, PI.getPersonId(), PI.getRolodexId(), COLLABORATOR_TYPE);
        if (collaboratorAttachment == null) {
            collaboratorAttachment = AttachedFileDataType.Factory.newInstance();
        }
        profile.getCollaboratorsAttached().setCollaboratorAttached(collaboratorAttachment);
    }

    private PersonProfileDataType[] getPersonProfileKeyPerson() {
        List<PersonProfileDataType> personProfileDataTypeList = new ArrayList<>();
        DevelopmentProposalContract developmentProposal = pdDoc.getDevelopmentProposal();
        List<? extends ProposalPersonContract> keyPersons = developmentProposal.getProposalPersons();
        if (keyPersons != null) {
            keyPersons.sort(new ProposalPersonComparator());
        }
        List<ProposalPersonContract> nKeyPersons = s2SProposalPersonService.getNKeyPersons(keyPersons, MAX_KEY_PERSON_COUNT);

        extraPersons = keyPersons != null ? keyPersons.stream()
                .filter(kp -> !nKeyPersons.contains(kp))
                .collect(Collectors.toList()) : Collections.emptyList();

        if (nKeyPersons.size() > 0) {
            setKeyPersonToPersonProfileDataType(personProfileDataTypeList, nKeyPersons);
        }

        return personProfileDataTypeList.toArray(new PersonProfileDataType[0]);
    }

    private void setKeyPersonToPersonProfileDataType(
            List<PersonProfileDataType> personProfileDataTypeList,
            List<ProposalPersonContract> nKeyPersons) {
        for (ProposalPersonContract keyPerson : nKeyPersons) {
            if (pIPersonOrRolodexId != null) {
                // Don't add PI to keyperson list
                if (keyPerson.getPersonId() != null && keyPerson.getPersonId().equals(pIPersonOrRolodexId)) {
                    continue;
                } else if ((keyPerson.getRolodexId() != null) && pIPersonOrRolodexId.equals(keyPerson.getRolodexId().toString())) {
                    continue;
                }
            }
            Profile profileKeyPerson = Profile.Factory.newInstance();
            setAllkeyPersonDetailsToKeyPerson(keyPerson, profileKeyPerson);
            setAttachments(profileKeyPerson, keyPerson);
            PersonProfileDataType personProfileDataTypeKeyPerson = PersonProfileDataType.Factory.newInstance();
            personProfileDataTypeKeyPerson.setProfile(profileKeyPerson);
            personProfileDataTypeList.add(personProfileDataTypeKeyPerson);
        }
    }

    private void setAllkeyPersonDetailsToKeyPerson(ProposalPersonContract keyPerson,
                                                   Profile profileKeyPerson) {
        assignRolodexId(keyPerson);
        profileKeyPerson.setName(globLibV20Generator.getHumanNameDataType(keyPerson));
        setDirectoryTitleToProfile(profileKeyPerson, keyPerson);
        profileKeyPerson.setAddress(globLibV20Generator.getAddressDataTypeV3(keyPerson));
        profileKeyPerson.setPhone(keyPerson.getOfficePhone());
        if (StringUtils.isNotEmpty(keyPerson.getFaxNumber())) {
            profileKeyPerson.setFax(keyPerson.getFaxNumber());
        }
        profileKeyPerson.setEmail(keyPerson.getEmailAddress());
        DevelopmentProposalContract developmentProposal = pdDoc.getDevelopmentProposal();
        setOrganizationName(profileKeyPerson, developmentProposal, keyPerson);
        setDepartmentNameToProfile(profileKeyPerson, keyPerson);
        final String divisionName = getS2sDivisionService().getDivision(keyPerson);
        if (divisionName != null) {
            profileKeyPerson.setDivisionName(divisionName);
        }
        profileKeyPerson.setNSFID(keyPerson.getNsfId() == null ? "" : keyPerson.getNsfId());
        setProjectRole(keyPerson, profileKeyPerson);
    }

    private void setProjectRole(ProposalPersonContract keyPerson, Profile profileKeyPerson) {
        if (keyPerson.isMultiplePi() || keyPerson.isCoInvestigator()) {
            if (getSponsorHierarchyService().isSponsorNihMultiplePi(pdDoc.getDevelopmentProposal().getSponsor().getSponsorCode())) {
                if (keyPerson.isMultiplePi()) {
                    profileKeyPerson.setProjectRole(ProjectRoleDataType.PD_PI);
                } else {
                    profileKeyPerson.setProjectRole(ProjectRoleDataType.CO_INVESTIGATOR);
                }
            } else {
                profileKeyPerson.setProjectRole(ProjectRoleDataType.CO_PD_PI);
            }
        } else {
            setProjectRoleCategoryToProfile(keyPerson, profileKeyPerson);
        }
    }

    private void setOrganizationName(Profile profileKeyPerson,
                                     DevelopmentProposalContract developmentProposal,
                                     ProposalPersonContract proposalPerson) {
        if (proposalPerson.getRolodexId() == null) {
            if (developmentProposal.getApplicantOrganization() != null) {
                profileKeyPerson.setOrganizationName(StringUtils.substring(developmentProposal.getApplicantOrganization().getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
            }
        } else {
            RolodexContract rolodex = rolodexService.getRolodex(proposalPerson.getRolodexId());

            if (rolodex != null && rolodex.getOrganization() != null) {
                profileKeyPerson.setOrganizationName(StringUtils.substring(rolodex.getOrganization(), 0, ORGANIZATON_NAME_MAX_LENGTH));
            }
        }
    }

    private void setProjectRoleCategoryToProfile(ProposalPersonContract keyPerson,
                                                 Profile profileKeyPerson) {
        if (keyPerson.getRolodexId() != null && keyPerson.getProjectRole().equals(ProjectRoleDataType.PD_PI.toString())) {
            profileKeyPerson.setProjectRole(ProjectRoleDataType.PD_PI);
        } else {
            profileKeyPerson.setProjectRole(ProjectRoleDataType.OTHER_SPECIFY);
            OtherProjectRoleCategory otherProjectRole = OtherProjectRoleCategory.Factory.newInstance();
            String otherRole;
            if (keyPerson.getProjectRole() != null) {
                if (keyPerson.getProjectRole().length() > ROLE_DESCRIPTION_MAX_LENGTH) {
                    otherRole = keyPerson.getProjectRole().substring(0, ROLE_DESCRIPTION_MAX_LENGTH);
                } else {
                    otherRole = keyPerson.getProjectRole();
                }
            } else {
                otherRole = FieldValueConstants.VALUE_UNKNOWN;
            }
            otherProjectRole.setStringValue(otherRole);
            profileKeyPerson.setOtherProjectRoleCategory(otherProjectRole);
        }
    }

    @Override
    public NSFKeyPersonExpanded20Document getFormObject(ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getNSFKeyPersonExpanded();
    }

    public UnitRepositoryService getUnitRepositoryService() {
        return unitRepositoryService;
    }

    public void setUnitRepositoryService(UnitRepositoryService unitRepositoryService) {
        this.unitRepositoryService = unitRepositoryService;
    }

    public RolodexService getRolodexService() {
        return rolodexService;
    }

    public void setRolodexService(RolodexService rolodexService) {
        this.rolodexService = rolodexService;
    }

    public S2SPrintingService getS2SPrintingService() {
        return s2SPrintingService;
    }

    public void setS2SPrintingService(S2SPrintingService s2SPrintingService) {
        this.s2SPrintingService = s2SPrintingService;
    }

    public S2SProposalPersonService getS2SProposalPersonService() {
        return s2SProposalPersonService;
    }

    public void setS2SProposalPersonService(S2SProposalPersonService s2SProposalPersonService) {
        this.s2SProposalPersonService = s2SProposalPersonService;
    }

    public S2sDivisionService getS2sDivisionService() {
        return s2sDivisionService;
    }

    public void setS2sDivisionService(S2sDivisionService s2sDivisionService) {
        this.s2sDivisionService = s2sDivisionService;
    }

    public Resource getAdditionalkeypersonprofilesStyleSheet() {
        return additionalkeypersonprofilesStyleSheet;
    }

    public void setAdditionalkeypersonprofilesStyleSheet(Resource additionalkeypersonprofilesStyleSheet) {
        this.additionalkeypersonprofilesStyleSheet = additionalkeypersonprofilesStyleSheet;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(NSFKeyPersonExpanded20Document form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {
            if (form.getNSFKeyPersonExpanded20().getKeyPersonList() != null) {
                final List<PersonProfileDataType> keyPersonList = form.getNSFKeyPersonExpanded20().getKeyPersonList();
                for (int i = 0; i < keyPersonList.size(); i++) {
                    final PersonProfileDataType personProfile = keyPersonList.get(i);
                    final PersonProfileDataType.Profile.BioSketchsAttached biosketch = personProfile.getProfile().getBioSketchsAttached();
                    final PersonProfileDataType.Profile.SupportsAttached supports = personProfile.getProfile().getSupportsAttached();
                    final PersonProfileDataType.Profile.CollaboratorsAttached collaborators = personProfile.getProfile().getCollaboratorsAttached();

                    if (biosketch != null && biosketch.getBioSketchAttached() != null && a.getContentId().equals(biosketch.getBioSketchAttached().getFileLocation().getHref())){
                        return entry("NSF_KeyPersonExpanded_2_0_P1.KeyPerson.optionalFile0_" + i, a);
                    }

                    if (supports != null && supports.getSupportAttached() != null && a.getContentId().equals(supports.getSupportAttached().getFileLocation().getHref())){
                        return entry("NSF_KeyPersonExpanded_2_0_P1.KeyPerson.optionalFile1_" + i, a);
                    }

                    if (collaborators != null && collaborators.getCollaboratorAttached() != null && a.getContentId().equals(collaborators.getCollaboratorAttached().getFileLocation().getHref())){
                        return entry("NSF_KeyPersonExpanded_2_0_P1.KeyPerson.mandatoryFile0_" + i, a);
                    }
                }
            }

            final NSFKeyPersonExpanded20.BioSketchsAttached bioSketchAttached = form.getNSFKeyPersonExpanded20().getBioSketchsAttached();
            if (bioSketchAttached != null && bioSketchAttached.getBioSketchAttached() != null && a.getContentId().equals(bioSketchAttached.getBioSketchAttached().getFileLocation().getHref())) {
                return entry("NSF_KeyPersonExpanded_2_0_P1.PDPI.mandatoryFile0", a);
            }

            final NSFKeyPersonExpanded20.SupportsAttached supportAttached = form.getNSFKeyPersonExpanded20().getSupportsAttached();
            if (supportAttached != null && supportAttached.getSupportAttached() != null && a.getContentId().equals(supportAttached.getSupportAttached().getFileLocation().getHref())) {
                return entry("NSF_KeyPersonExpanded_2_0_P1.PDPI.optionalFile0", a);
            }

            final NSFKeyPersonExpanded20.CollaboratorsAttached collaboratorAttached = form.getNSFKeyPersonExpanded20().getCollaboratorsAttached();
            if (collaboratorAttached != null && collaboratorAttached.getCollaboratorAttached() != null && a.getContentId().equals(collaboratorAttached.getCollaboratorAttached().getFileLocation().getHref())) {
                return entry("NSF_KeyPersonExpanded_2_0_P1.PDPI.mandatoryFile1", a);
            }

            final AdditionalProfilesAttached additionalProfileAttached = form.getNSFKeyPersonExpanded20().getAdditionalProfilesAttached();
            if (additionalProfileAttached != null && additionalProfileAttached.getAdditionalProfileAttached() != null && a.getContentId().equals(additionalProfileAttached.getAdditionalProfileAttached().getFileLocation().getHref())) {
                return entry("NSF_KeyPersonExpanded_2_0_P1.optionalFile1", a);
            }

            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<NSFKeyPersonExpanded20Document> factory() {
        return NSFKeyPersonExpanded20Document.Factory;
    }
}
