/*-
 * #%L
 * %%
 * Copyright (C) 2014 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.coeus.s2sgen.impl.generate.support;

import gov.grants.apply.forms.rrKeyPersonExpanded40V40.PersonProfileDataType;
import gov.grants.apply.forms.rrKeyPersonExpanded40V40.PersonProfileDataType.Profile;
import gov.grants.apply.forms.rrKeyPersonExpanded40V40.PersonProfileDataType.Profile.OtherProjectRoleCategory;
import gov.grants.apply.forms.rrKeyPersonExpanded40V40.ProjectRoleDataType;
import gov.grants.apply.forms.rrKeyPersonExpanded40V40.RRKeyPersonExpanded40Document;
import gov.grants.apply.forms.rrKeyPersonExpanded40V40.RRKeyPersonExpanded40Document.RRKeyPersonExpanded40;
import gov.grants.apply.forms.rrKeyPersonExpanded40V40.RRKeyPersonExpanded40Document.RRKeyPersonExpanded40.AdditionalProfilesAttached;
import gov.grants.apply.forms.rrKeyPersonExpanded40V40.RRKeyPersonExpanded40Document.RRKeyPersonExpanded40.BioSketchsAttached;
import gov.grants.apply.forms.rrKeyPersonExpanded40V40.RRKeyPersonExpanded40Document.RRKeyPersonExpanded40.SupportsAttached;
import gov.grants.apply.system.attachmentsV10.AttachedFileDataType;
import org.apache.commons.lang3.StringUtils;
import org.apache.xmlbeans.impl.schema.DocumentFactory;
import org.kuali.coeus.common.api.rolodex.RolodexContract;
import org.kuali.coeus.common.api.rolodex.RolodexService;
import org.kuali.coeus.propdev.api.attachment.NarrativeContract;
import org.kuali.coeus.propdev.api.core.DevelopmentProposalContract;
import org.kuali.coeus.propdev.api.core.ProposalDevelopmentDocumentContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonContract;
import org.kuali.coeus.propdev.api.person.ProposalPersonDegreeContract;
import org.kuali.coeus.s2sgen.api.core.AuditError;
import org.kuali.coeus.s2sgen.api.generate.AttachmentData;
import org.kuali.coeus.s2sgen.impl.generate.FormGenerator;
import org.kuali.coeus.s2sgen.impl.generate.FormStylesheet;
import org.kuali.coeus.s2sgen.impl.generate.FormVersion;
import org.kuali.coeus.s2sgen.impl.generate.S2SFormGeneratorPdfFillable;
import org.kuali.coeus.s2sgen.impl.util.FieldValueConstants;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.Resource;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entriesToMap;
import static org.kuali.coeus.s2sgen.impl.util.CollectionUtils.entry;

@FormGenerator("RRKeyPersonExpandedV4_0Generator")
public class RRKeyPersonExpandedV4_0Generator extends
        RRKeyPersonExpandedBaseGenerator<RRKeyPersonExpanded40Document> implements S2SFormGeneratorPdfFillable<RRKeyPersonExpanded40Document> {

    private static final int MAX_KEY_PERSON_COUNT = 100;
    private static final String CHILD = "C";
    private static final int DEPARTMENT_NAME_MAX_LENGTH = 100;

    RolodexContract rolodex;

    @Value("http://apply.grants.gov/forms/RR_KeyPersonExpanded_4_0-V4.0")
    private String namespace;

    @Value("RR_KeyPersonExpanded_4_0")
    private String formName;

    @FormStylesheet
    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/stylesheet/RR_KeyPersonExpanded-V4.0.xsl")
    private List<Resource> stylesheets;

    @Value("classpath:org/kuali/coeus/s2sgen/impl/generate/support/pdf/RR_KeyPersonExpanded_4_0-V4.0.pdf")
    private Resource pdfForm;

    @Value("155")
    private int sortIndex;

    @Autowired
    @Qualifier("rolodexService")
    private RolodexService rolodexService;

    private RRKeyPersonExpanded40Document getRRKeyPersonExpanded() {
        RRKeyPersonExpanded40Document rrKeyPersonExpandedDocument = RRKeyPersonExpanded40Document.Factory
                .newInstance();
        RRKeyPersonExpanded40 rrKeyPersonExpanded = RRKeyPersonExpanded40.Factory
                .newInstance();
        setRRKeyPersonExpandedAttributes(rrKeyPersonExpanded);
        rrKeyPersonExpandedDocument
                .setRRKeyPersonExpanded40(rrKeyPersonExpanded);
        return rrKeyPersonExpandedDocument;
    }

    private void setRRKeyPersonExpandedAttributes(
            RRKeyPersonExpanded40 rrKeyPersonExpanded) {
        rrKeyPersonExpanded.setFormVersion(FormVersion.v4_0.getVersion());
        rrKeyPersonExpanded.setPDPI(getPersonProfilePI());
        PersonProfileDataType[] keyPersonArray = getPersonProfileKeyPerson();
        if (keyPersonArray.length > 0) {
            rrKeyPersonExpanded.setKeyPersonArray(keyPersonArray);
        }
        saveKeyPersonAttachmentsToProposal();
        if (extraPersons.size() > 0) {
            for (ProposalPersonContract extraPerson : extraPersons) {
                setBioSketchAttchment(rrKeyPersonExpanded, extraPerson);
                setCurrentPendingTypeAttachment(rrKeyPersonExpanded,
                        extraPerson);
            }
            for (NarrativeContract narrative : pdDoc.getDevelopmentProposal()
                    .getNarratives()) {
                if (narrative.getNarrativeType().getCode() != null) {
                    if (Integer.parseInt(narrative.getNarrativeType().getCode()) == PROFILE_TYPE) {
                        setProfileTypeAttachment(rrKeyPersonExpanded, narrative);
                    }
                }
            }
        }
    }

    private void setProfileTypeAttachment(
            RRKeyPersonExpanded40 rrKeyPersonExpanded, NarrativeContract narrative) {
        AttachedFileDataType attachedFileDataType = addAttachedFileType(narrative);
        if(attachedFileDataType != null){
            AdditionalProfilesAttached additionalProfilesAttached = AdditionalProfilesAttached.Factory
                    .newInstance();
            additionalProfilesAttached
                    .setAdditionalProfileAttached(attachedFileDataType);
            rrKeyPersonExpanded
                    .setAdditionalProfilesAttached(additionalProfilesAttached);
        }
    }

    private void setCurrentPendingTypeAttachment(
            RRKeyPersonExpanded40 rrKeyPersonExpanded,
            ProposalPersonContract extraPerson) {
        AttachedFileDataType supportAttachment = getPersonnelAttachments(pdDoc,
                extraPerson.getPersonId(), extraPerson.getRolodexId(),
                CURRENT_PENDING_TYPE);
        if (supportAttachment != null) {
            SupportsAttached supportsAttached = SupportsAttached.Factory
                    .newInstance();
            supportsAttached.setSupportAttached(supportAttachment);
            rrKeyPersonExpanded.setSupportsAttached(supportsAttached);
        }
    }

    private void setBioSketchAttchment(
            RRKeyPersonExpanded40 rrKeyPersonExpanded,
            ProposalPersonContract extraPerson) {
        BioSketchsAttached
                personBioSketch =  BioSketchsAttached.Factory.newInstance();
        AttachedFileDataType bioSketchAttachment = getPersonnelAttachments(
                pdDoc, extraPerson.getPersonId(), extraPerson.getRolodexId(),
                BIOSKETCH_TYPE);
        personBioSketch.setBioSketchAttached(bioSketchAttachment);
        rrKeyPersonExpanded.setBioSketchsAttached(personBioSketch);
    }

    private PersonProfileDataType getPersonProfilePI() {
        PersonProfileDataType profileDataType = PersonProfileDataType.Factory
                .newInstance();
        Profile profile = Profile.Factory.newInstance();
        ProposalPersonContract PI = s2SProposalPersonService.getPrincipalInvestigator(pdDoc);
        if (PI != null) {
            setPersonalProfileDetailsToProfile(profileDataType, profile, PI);
        }
        return profileDataType;
    }

    private void setPersonalProfileDetailsToProfile(
            PersonProfileDataType profileDataType, Profile profile,
            ProposalPersonContract PI) {
        assignRolodexId(PI);
        profile.setName(globLibV20Generator.getHumanNameDataType(PI));
        setDirectoryTitleToProfile(profile, PI);
        profile.setAddress(globLibV20Generator.getAddressDataTypeV3(PI));
        profile.setPhone(PI.getOfficePhone());
        if (StringUtils.isNotEmpty(PI.getFaxNumber())) {
            profile.setFax(PI.getFaxNumber());
        }
        if (PI.getDegree() != null) {
            profile.setDegreeType(PI.getDegree());
        }
        if (PI.getYearGraduated() != null) {
            profile.setDegreeYear(PI.getYearGraduated());
        }
        if(PI.getDegree() == null && PI.getYearGraduated() == null ){
            if(PI.getProposalPersonDegrees() != null && PI.getProposalPersonDegrees().size() > 0){
                ProposalPersonDegreeContract proposalPersonDegree = PI.getProposalPersonDegrees().get(0);
                if(proposalPersonDegree != null){
                    if(proposalPersonDegree.getDegreeType() != null && proposalPersonDegree.getDegreeType().getDescription()!= null)
                        profile.setDegreeType(proposalPersonDegree.getDegreeType().getDescription());
                    if(proposalPersonDegree.getGraduationYear() != null)
                        profile.setDegreeYear(proposalPersonDegree.getGraduationYear());
                }
            }
        }
        profile.setEmail(PI.getEmailAddress());
        DevelopmentProposalContract developmentProposal = pdDoc
                .getDevelopmentProposal();
        setOrganizationName(profile, developmentProposal);
        setDepartmentNameToProfile(profile,PI);
        setDivisionNameToProfile(profile, PI);
        if (PI.getEraCommonsUserName() != null) {
            profile.setCredential(PI.getEraCommonsUserName());
        } else {
            if (getSponsorHierarchyService().isSponsorNihMultiplePi(pdDoc.getDevelopmentProposal().getSponsor().getSponsorCode())) {
                getAuditErrors().add(new AuditError(AuditError.NO_FIELD_ERROR_KEY, ERROR_ERA_COMMON_USER_NAME + PI.getFullName(),
                        AuditError.GG_LINK));
            }
        }

        //proposal hierarchy specific alteration.. PI becomes project lead in children
        if (CHILD.equalsIgnoreCase(developmentProposal.getHierarchyStatus())) {
            profile.setProjectRole(ProjectRoleDataType.OTHER_SPECIFY);
            OtherProjectRoleCategory otherProjectRoleCategory = OtherProjectRoleCategory.Factory.newInstance();
            otherProjectRoleCategory.setStringValue("Project Lead");
            profile.setOtherProjectRoleCategory(otherProjectRoleCategory);
        } else {
            profile.setProjectRole(ProjectRoleDataType.PD_PI);
        }

        setAttachments(profile, PI, true);
        profileDataType.setProfile(profile);
    }

    private void setDivisionNameToProfile(Profile profile, ProposalPersonContract pi) {
        final String divisionName = getS2sDivisionService().getDivision(pi);
        if (divisionName != null) {
            profile.setDivision(divisionName);
        }
    }

    private void setDepartmentNameToProfile(Profile profile, ProposalPersonContract keyPerson) {
        if(keyPerson.getHomeUnit() != null && keyPerson.getPerson() != null && keyPerson.getPerson().getUnit() != null) {
            final String departmentName =  keyPerson.getPerson().getUnit().getUnitName();
            profile.setDepartment(StringUtils.substring(departmentName, 0, DEPARTMENT_NAME_MAX_LENGTH));
        }
        else
        {
            if (keyPerson.getRolodexId() != null && keyPerson.getDirectoryDepartment() != null) {
                // If this is a non-employee (rolodex not null), pull the Directory Department field
                // from the person if a value has been entered there
                profile.setDepartment(StringUtils.substring(keyPerson.getDirectoryDepartment(), 0, DEPARTMENT_NAME_MAX_LENGTH));
            } else {
                DevelopmentProposalContract developmentProposal = pdDoc.getDevelopmentProposal();
                profile.setDepartment(StringUtils.substring(developmentProposal.getOwnedByUnit().getUnitName(), 0, DEPARTMENT_NAME_MAX_LENGTH));
            }
        }
    }

    private void setDirectoryTitleToProfile(Profile profile, ProposalPersonContract PI) {
        if (PI.getDirectoryTitle() != null) {
            if (PI.getDirectoryTitle().length() > DIRECTORY_TITLE_MAX_LENGTH) {
                profile.setTitle(PI.getDirectoryTitle().substring(0,
                        DIRECTORY_TITLE_MAX_LENGTH));
            } else {
                profile.setTitle(PI.getDirectoryTitle());
            }
        }
    }

    private void assignRolodexId(ProposalPersonContract PI) {
        if (PI.getPersonId() != null) {
            pIPersonOrRolodexId = PI.getPersonId();
            rolodex = null;
        } else if (PI.getRolodexId() != null) {
            pIPersonOrRolodexId = PI.getRolodexId().toString();
            rolodex = rolodexService.getRolodex(Integer.valueOf(pIPersonOrRolodexId));
        }
    }

    private void setAttachments(Profile profile, ProposalPersonContract PI, boolean biosketchRequired) {
        setBioSketchAttachment(profile, PI, biosketchRequired);
        setCurrentPendingAttachment(profile, PI);
    }

    private void setCurrentPendingAttachment(Profile profile, ProposalPersonContract PI) {
        AttachedFileDataType supportAttachment = getPersonnelAttachments(pdDoc,
                PI.getPersonId(), PI.getRolodexId(), CURRENT_PENDING_TYPE);
        if (supportAttachment != null) {
            Profile.SupportsAttached supportsAttached = Profile.SupportsAttached.Factory
                    .newInstance();
            supportsAttached.setSupportAttached(supportAttachment);
            profile.setSupportsAttached(supportsAttached);
        }
    }

    private void setBioSketchAttachment(Profile profile, ProposalPersonContract PI, boolean biosketchRequired) {
        AttachedFileDataType bioSketchAttachment = getPersonnelAttachments(
                pdDoc, PI.getPersonId(), PI.getRolodexId(), BIOSKETCH_TYPE);
        if (bioSketchAttachment != null) {
            Profile.BioSketchsAttached personBioSketch = Profile.BioSketchsAttached.Factory
                    .newInstance();
            personBioSketch.setBioSketchAttached(bioSketchAttachment);
            profile.setBioSketchsAttached(personBioSketch);
        } else if (biosketchRequired) {
            Profile.BioSketchsAttached personBioSketch = Profile.BioSketchsAttached.Factory
                    .newInstance();
            personBioSketch.setBioSketchAttached(AttachedFileDataType.Factory.newInstance());
            profile.setBioSketchsAttached(personBioSketch);
        }
    }

    private PersonProfileDataType[] getPersonProfileKeyPerson() {
        List<PersonProfileDataType> personProfileDataTypeList = new ArrayList<>();
        DevelopmentProposalContract developmentProposal = pdDoc
                .getDevelopmentProposal();
        List<? extends ProposalPersonContract> keyPersons = developmentProposal
                .getProposalPersons();
        if (keyPersons != null) {
            keyPersons.sort(new ProposalPersonComparator());
        }
        List<ProposalPersonContract> nKeyPersons = s2SProposalPersonService.getNKeyPersons(
                keyPersons, MAX_KEY_PERSON_COUNT);
        extraPersons = keyPersons != null ? keyPersons.stream()
                .filter(kp -> !nKeyPersons.contains(kp))
                .collect(Collectors.toList()) : Collections.emptyList();
        if (nKeyPersons.size() > 0) {
            setKeyPersonToPersonProfileDataType(personProfileDataTypeList,
                    nKeyPersons);
        }
        return personProfileDataTypeList.toArray(new PersonProfileDataType[0]);
    }

    private void setKeyPersonToPersonProfileDataType(
            List<PersonProfileDataType> personProfileDataTypeList,
            List<ProposalPersonContract> nKeyPersons) {
        for (ProposalPersonContract keyPerson : nKeyPersons) {
            if (pIPersonOrRolodexId != null) {
                // Don't add PI to keyperson list
                if (keyPerson.getPersonId() != null
                        && keyPerson.getPersonId().equals(pIPersonOrRolodexId)) {
                    continue;
                } else if ((keyPerson.getRolodexId() != null)
                        && pIPersonOrRolodexId.equals(keyPerson.getRolodexId()
                        .toString())) {
                    continue;
                }
            }
            Profile profileKeyPerson = Profile.Factory.newInstance();
            setAllkeyPersonDetailsToKeyPerson(keyPerson, profileKeyPerson);
            setAttachments(profileKeyPerson, keyPerson, false);
            PersonProfileDataType personProfileDataTypeKeyPerson = PersonProfileDataType.Factory
                    .newInstance();
            personProfileDataTypeKeyPerson.setProfile(profileKeyPerson);
            personProfileDataTypeList.add(personProfileDataTypeKeyPerson);
        }
    }

    /*
     * This method is used to add all key person details to key person
     */
    private void setAllkeyPersonDetailsToKeyPerson(ProposalPersonContract keyPerson,
                                                   Profile profileKeyPerson) {
        assignRolodexId(keyPerson);
        profileKeyPerson.setName(globLibV20Generator
                .getHumanNameDataType(keyPerson));
        setDirectoryTitleToProfile(profileKeyPerson, keyPerson);
        profileKeyPerson.setAddress(globLibV20Generator
                .getAddressDataTypeV3(keyPerson));
        profileKeyPerson.setPhone(keyPerson.getOfficePhone());
        if (StringUtils.isNotEmpty(keyPerson.getFaxNumber())) {
            profileKeyPerson.setFax(keyPerson.getFaxNumber());
        }
        if (keyPerson.getDegree() != null) {
            profileKeyPerson.setDegreeType(keyPerson.getDegree());
        }
        if (keyPerson.getYearGraduated() != null) {
            profileKeyPerson.setDegreeYear(keyPerson.getYearGraduated());
        }
        if(keyPerson.getDegree() == null && keyPerson.getYearGraduated() == null ){
            if(keyPerson.getProposalPersonDegrees() != null && keyPerson.getProposalPersonDegrees().size() > 0){
                ProposalPersonDegreeContract proposalPersonDegree = keyPerson.getProposalPersonDegrees().get(0);
                if(proposalPersonDegree != null){
                    if(proposalPersonDegree.getDegreeType() != null &&proposalPersonDegree.getDegreeType().getDescription() != null)
                        profileKeyPerson.setDegreeType(proposalPersonDegree.getDegreeType().getDescription());
                    if(proposalPersonDegree.getGraduationYear() != null)
                        profileKeyPerson.setDegreeYear(proposalPersonDegree.getGraduationYear());
                }
            }
        }
        profileKeyPerson.setEmail(keyPerson.getEmailAddress());
        DevelopmentProposalContract developmentProposal = pdDoc
                .getDevelopmentProposal();
        setOrganizationName(profileKeyPerson, developmentProposal);
        setDepartmentNameToProfile(profileKeyPerson,keyPerson);
        setDivisionNameToProfile(profileKeyPerson, keyPerson);
        if (keyPerson.getEraCommonsUserName() != null) {
            profileKeyPerson.setCredential(keyPerson.getEraCommonsUserName());
        } else {
            if (getSponsorHierarchyService().isSponsorNihMultiplePi(pdDoc.getDevelopmentProposal().getSponsor().getSponsorCode())) {
                if (keyPerson.isMultiplePi()) {
                    getAuditErrors().add(new AuditError(AuditError.NO_FIELD_ERROR_KEY, ERROR_ERA_COMMON_USER_NAME + keyPerson.getFullName(),
                            AuditError.GG_LINK));
                }
            }
        }
        if (keyPerson.isMultiplePi() || keyPerson.isCoInvestigator()) {
            if(getSponsorHierarchyService().isSponsorNihMultiplePi(pdDoc.getDevelopmentProposal().getSponsor().getSponsorCode())){
                if (keyPerson.isMultiplePi()) {
                    profileKeyPerson.setProjectRole(ProjectRoleDataType.PD_PI);
                } else {
                    profileKeyPerson.setProjectRole(ProjectRoleDataType.CO_INVESTIGATOR);
                }
            }else{
                profileKeyPerson.setProjectRole(ProjectRoleDataType.CO_PD_PI);
            }
        } else {
            setProjectRoleCategoryToProfile(keyPerson, profileKeyPerson);
        }
    }

    private void setOrganizationName(Profile profileKeyPerson,
                                     DevelopmentProposalContract developmentProposal) {
        if (developmentProposal.getApplicantOrganization() != null) {
            if (rolodex != null) {
                profileKeyPerson.setOrganizationName(StringUtils.substring(rolodex.getOrganization(), 0, ORGANIZATON_NAME_MAX_LENGTH));
            } else {
                profileKeyPerson.setOrganizationName(StringUtils.substring(pdDoc.getDevelopmentProposal().getApplicantOrganization().getLocationName(), 0, ORGANIZATON_NAME_MAX_LENGTH));
            }
        }
    }

    private void setProjectRoleCategoryToProfile(ProposalPersonContract keyPerson,
                                                 Profile profileKeyPerson) {
        if (keyPerson.getRolodexId() != null
                && StringUtils.equalsIgnoreCase(keyPerson.getProjectRole(), ProjectRoleDataType.PD_PI.toString())) {
            profileKeyPerson.setProjectRole(ProjectRoleDataType.PD_PI);
        } else {
            profileKeyPerson.setProjectRole(ProjectRoleDataType.OTHER_SPECIFY);
            OtherProjectRoleCategory otherProjectRole = OtherProjectRoleCategory.Factory
                    .newInstance();
            String otherRole;
            if (keyPerson.getProjectRole() != null) {
                if (keyPerson.getProjectRole().length() > ROLE_DESCRIPTION_MAX_LENGTH) {
                    otherRole = keyPerson.getProjectRole().substring(0,
                            ROLE_DESCRIPTION_MAX_LENGTH);
                } else {
                    otherRole = keyPerson.getProjectRole();
                }
            } else {
                otherRole = FieldValueConstants.VALUE_UNKNOWN;
            }
            otherProjectRole.setStringValue(otherRole);
            profileKeyPerson.setOtherProjectRoleCategory(otherProjectRole);
        }
    }

    @Override
    public RRKeyPersonExpanded40Document getFormObject(
            ProposalDevelopmentDocumentContract proposalDevelopmentDocument) {
        this.pdDoc = proposalDevelopmentDocument;
        return getRRKeyPersonExpanded();
    }

    public RolodexService getRolodexService() {
        return rolodexService;
    }

    public void setRolodexService(RolodexService rolodexService) {
        this.rolodexService = rolodexService;
    }

    @Override
    public String getNamespace() {
        return namespace;
    }

    public void setNamespace(String namespace) {
        this.namespace = namespace;
    }

    @Override
    public String getFormName() {
        return formName;
    }

    public void setFormName(String formName) {
        this.formName = formName;
    }

    @Override
    public List<Resource> getStylesheets() {
        return stylesheets;
    }

    public void setStylesheets(List<Resource> stylesheets) {
        this.stylesheets = stylesheets;
    }

    @Override
    public Resource getPdfForm() {
        return pdfForm;
    }

    public void setPdfForm(Resource pdfForm) {
        this.pdfForm = pdfForm;
    }

    @Override
    public int getSortIndex() {
        return sortIndex;
    }

    public void setSortIndex(int sortIndex) {
        this.sortIndex = sortIndex;
    }

    @Override
    public Attachments getMappedAttachments(RRKeyPersonExpanded40Document form, List<AttachmentData> attachments) {
        final Map<Boolean, List<Map.Entry<String, AttachmentData>>> attachmentPartition = attachments.stream().map(a -> {

            final AdditionalProfilesAttached additionalProfilesAttached = form.getRRKeyPersonExpanded40().getAdditionalProfilesAttached();
            if (additionalProfilesAttached != null && additionalProfilesAttached.getAdditionalProfileAttached() != null && a.getContentId().equals(additionalProfilesAttached.getAdditionalProfileAttached().getFileLocation().getHref())) {
                return entry("RR_KeyPersonExpanded_4_0_P1.optionalFile1", a);
            }

            final BioSketchsAttached bioSketchsAttached = form.getRRKeyPersonExpanded40().getBioSketchsAttached();
            if (bioSketchsAttached != null && bioSketchsAttached.getBioSketchAttached() != null && a.getContentId().equals(bioSketchsAttached.getBioSketchAttached().getFileLocation().getHref())) {
                return entry("RR_KeyPersonExpanded_4_0_P1.optionalFile2", a);
            }

            final SupportsAttached supportsAttached = form.getRRKeyPersonExpanded40().getSupportsAttached();
            if (supportsAttached != null && supportsAttached.getSupportAttached() != null && a.getContentId().equals(supportsAttached.getSupportAttached().getFileLocation().getHref())) {
                return entry("RR_KeyPersonExpanded_4_0_P1.optionalFile3", a);
            }

            final PersonProfileDataType pdpi = form.getRRKeyPersonExpanded40().getPDPI();
            if (pdpi != null && pdpi.getProfile() != null) {
                final PersonProfileDataType.Profile.BioSketchsAttached pdpiBioSketchsAttached = pdpi.getProfile().getBioSketchsAttached();
                if (pdpiBioSketchsAttached != null && pdpiBioSketchsAttached.getBioSketchAttached() != null && a.getContentId().equals(pdpiBioSketchsAttached.getBioSketchAttached().getFileLocation().getHref())) {
                    return entry("RR_KeyPersonExpanded_4_0_P1.PDPI.mandatoryFile0", a);
                }

                final PersonProfileDataType.Profile.SupportsAttached pdpiSupportsAttached = pdpi.getProfile().getSupportsAttached();
                if (pdpiSupportsAttached != null && pdpiSupportsAttached.getSupportAttached() != null && a.getContentId().equals(pdpiSupportsAttached.getSupportAttached().getFileLocation().getHref())) {
                    return entry("RR_KeyPersonExpanded_4_0_P1.PDPI.optionalFile0", a);
                }
            }

            final List<PersonProfileDataType> keyPersons = form.getRRKeyPersonExpanded40().getKeyPersonList();
            if (keyPersons != null) {
                for (int i = 0; i < keyPersons.size(); i++) {
                    final PersonProfileDataType keyPerson = keyPersons.get(i);
                    if (keyPerson != null && keyPerson.getProfile() != null) {
                        final PersonProfileDataType.Profile.BioSketchsAttached keyPersonBioSketchsAttached = keyPerson.getProfile().getBioSketchsAttached();
                        if (keyPersonBioSketchsAttached != null && keyPersonBioSketchsAttached.getBioSketchAttached() != null && a.getContentId().equals(keyPersonBioSketchsAttached.getBioSketchAttached().getFileLocation().getHref())) {
                            return entry("RR_KeyPersonExpanded_4_0_P1.KeyPerson.optionalFile0_" + i, a);
                        }

                        final PersonProfileDataType.Profile.SupportsAttached keyPersonSupportsAttached = keyPerson.getProfile().getSupportsAttached();
                        if (keyPersonSupportsAttached != null && keyPersonSupportsAttached.getSupportAttached() != null && a.getContentId().equals(keyPersonSupportsAttached.getSupportAttached().getFileLocation().getHref())) {
                            return entry("RR_KeyPersonExpanded_4_0_P1.KeyPerson.optionalFile1_" + i, a);
                        }
                    }
                }
            }
            return entry((String) null, a);
        }).collect(Collectors.partitioningBy(a -> StringUtils.isNotBlank(a.getKey())));

        return new Attachments(attachmentPartition.get(Boolean.TRUE).stream().collect(entriesToMap()),
                attachmentPartition.get(Boolean.FALSE).stream().map(Map.Entry::getValue).collect(Collectors.toList()));
    }

    @Override
    public DocumentFactory<RRKeyPersonExpanded40Document> factory() {
        return RRKeyPersonExpanded40Document.Factory;
    }
}
