/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.util;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.config.property.ConfigStrLookup;
import org.kuali.rice.coreservice.api.CoreServiceApiServiceLocator;
import org.kuali.rice.coreservice.api.parameter.ParameterKey;
import org.kuali.rice.coreservice.framework.CoreFrameworkServiceLocator;
import org.kuali.rice.kew.api.KewApiConstants;
import org.kuali.rice.krad.util.KRADConstants;

/**
 * Uses the KEW runtime parameters to locate a string for replacement, falling back to the deploy time configuration
 * variables if necessary.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class ParameterStrLookup extends ConfigStrLookup {
	
	private final String applicationId;

    /**
     * Creates a string locator to search for KEW runtime parameters for any {@code applicationId}.
     */
	public ParameterStrLookup() {
        this(null);
	}

    /**
     * Creates a string locator to search for KEW runtime parameters for a specific {@code applicationId}.
     *
     * @param applicationId the application to search for the KEW runtime parameter in.
     */
	public ParameterStrLookup(String applicationId) {
		this.applicationId = applicationId;
	}

    /**
     * {@inheritDoc}
     */
    @Override
    public String lookup(String propertyName) {
        if (StringUtils.isBlank(propertyName)) {
            return null;
        }

        String paramValue;

        // check runtime parameters first
        if (StringUtils.isBlank(applicationId)) {
            paramValue = CoreFrameworkServiceLocator.getParameterService().getParameterValueAsString(
                    KewApiConstants.KEW_NAMESPACE, KRADConstants.DetailTypes.ALL_DETAIL_TYPE, propertyName);
        } else {
            ParameterKey parameterKey = ParameterKey.create(applicationId, KewApiConstants.KEW_NAMESPACE,
                    KRADConstants.DetailTypes.ALL_DETAIL_TYPE, propertyName);
            paramValue = CoreServiceApiServiceLocator.getParameterRepositoryService().getParameterValueAsString(parameterKey);
        }

        // fall back to deploy time configurations if empty
        if (paramValue == null) {
            paramValue = super.lookup(propertyName);
        }

        return paramValue;
    }

}
