/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kcb.web.spring;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.kcb.bo.RecipientDelivererConfig;
import org.kuali.rice.kcb.deliverer.MessageDeliverer;
import org.kuali.rice.kcb.exception.ErrorList;
import org.kuali.rice.kcb.service.KENIntegrationService;
import org.kuali.rice.kcb.service.MessageDelivererRegistryService;
import org.kuali.rice.kcb.service.RecipientPreferenceService;
import org.kuali.rice.kns.util.WebUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

/**
 * This class is the controller that handles management of various user preferences interfaces (deliver types, user subscriptions, etc).
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@RequestMapping(value="/kcb")
@Controller("prefsController")
public class UserPreferencesController {
    /** Logger for this class and subclasses */
    private static final Logger LOG = LogManager.getLogger(UserPreferencesController.class);

    private static final String VIEW = "DelivererPreferences";

    @Autowired
    @Qualifier("recipientPreferenceService")
    protected RecipientPreferenceService recipientPreferenceService;

    @Autowired
    @Qualifier("messageDelivererRegistryService")
    protected MessageDelivererRegistryService messageDelivererRegistryService;

    @Autowired
    @Qualifier("kenIntegrationService")
    protected KENIntegrationService kenIntegrationService;

    /**
     * @return all channels for Rice, including the builtin KEW action list "channel"
     */
    protected Collection<String> getAllChannels() {
        // TODO: does not traverse bus yet
        Collection<String> allChannels = new ArrayList<String>();
        //allChannels.add(KEW_CHANNEL);
        allChannels.addAll(kenIntegrationService.getAllChannelNames());
        return allChannels;
    }

    /**
     * displayDelivererConfigurationForm - obtain information necessary
     * for displaying all possible Deliverer types and forward to the form
     */
    @RequestMapping("/prefs.kcb")
    public ModelAndView displayDelivererConfigurationForm(HttpServletRequest request) {
        String userid = request.getRemoteUser();
        LOG.debug("remoteUser: "+userid); 

        // Get DeliveryType classes
        Collection<MessageDeliverer> deliveryTypes = this.messageDelivererRegistryService.getAllDeliverers();

        // get all channels       
        Collection<String> channels = getAllChannels();

        //     get all user preferences in a HashMap
        HashMap<String, String> preferences  = this.recipientPreferenceService.getRecipientPreferences(userid);

        // get existing configured deliverers
        Collection<RecipientDelivererConfig> currentDeliverers = this.recipientPreferenceService.getDeliverersForRecipient(userid);
        // create a Map as an easy way for the JSP to determine whether a deliver is enabled for channels
        Map<String, Boolean> currentDeliverersMap = new HashMap<String, Boolean>();
        for (RecipientDelivererConfig udc: currentDeliverers) {
            String channelName = udc.getChannel();
            currentDeliverersMap.put(udc.getDelivererName() + "." + channelName, Boolean.TRUE);
        }

        Map<String, Object> model = new HashMap<String, Object>();
        model.put("channels", channels);
        model.put("deliveryTypes", deliveryTypes);
        model.put("preferences", preferences);
        model.put("currentDeliverersMap", currentDeliverersMap);
        putBackLocation(model, request.getParameter("backLocation"));

        return new ModelAndView(VIEW, model);
    }

    private void putBackLocation(Map<String, Object> model, String backLocation) {
        if (StringUtils.isNotBlank(backLocation))   {
            model.put("backLocation", WebUtils.sanitizeBackLocation(backLocation));
        }
    }

    /**
     * saveDelivererConfiguration - save deliverer configuration data
     */
    @RequestMapping("/saveprefs.kcb")
    public ModelAndView saveDelivererConfiguration(HttpServletRequest request) {
        String userid = request.getRemoteUser();
        LOG.debug("remoteUser: "+userid);
        boolean error = false;

        Map<String, Object> model = new HashMap<String, Object>();

        // create preferences map here so that we can pass them all back to the view
        HashMap<String, String> preferences  = new HashMap<String, String>();

        // Get DeliveryType classes.  loop through each deliverer type to 
        // to obtain preferenceKeys.  Check to see if a matching request
        // parameter was provided, then save a record for the userID, channelID, and 
        // preferences setting
        Collection<MessageDeliverer> deliveryTypes = this.messageDelivererRegistryService.getAllDeliverers();

        // first remove all configured user delivers for this user
        this.recipientPreferenceService.removeRecipientDelivererConfigs(userid);        

        for (MessageDeliverer dt: deliveryTypes) {
            String deliveryTypeName = dt.getName();
            HashMap<String,String> prefMap = dt.getPreferenceKeys();
            LOG.debug("deliveryName: "+deliveryTypeName);
            HashMap<String, String> userprefs = new HashMap<String, String>();
            for (String prefKey:prefMap.keySet()) {
                LOG.debug("   key: "+prefKey+", value: "+request.getParameter(deliveryTypeName+"."+prefKey));
                userprefs.put(deliveryTypeName+"."+prefKey, request.getParameter(deliveryTypeName+"."+prefKey ));
                preferences.put(deliveryTypeName+"."+prefKey, request.getParameter(deliveryTypeName+"."+prefKey ));
            }
            try {
                this.recipientPreferenceService.saveRecipientPreferences(userid, userprefs, dt);
            } catch (ErrorList errorlist) {
                error = true;
                model.put("errorList", errorlist.getErrors()) ;
            }

            // get channelName.channels
            String[] channels = request.getParameterValues(deliveryTypeName+".channels");
            if (channels != null && channels.length > 0) {
                for (int j=0; j < channels.length; j++) {
                    LOG.debug(deliveryTypeName+".channels["+j+"] "+channels[j]);   
                }
            }
            //	 now save the userid, channel selection
            this.recipientPreferenceService.saveRecipientDelivererConfig(userid, deliveryTypeName, channels);
        }

        // get all channels       
        Collection<String> channels = getAllChannels();

        // get existing configured deliverers
        Collection<RecipientDelivererConfig> currentDeliverers = this.recipientPreferenceService.getDeliverersForRecipient(userid);
        Map<String, Object> currentDeliverersMap = new HashMap<String, Object>();
        for (RecipientDelivererConfig udc: currentDeliverers) {
            String channelId = udc.getChannel();
            currentDeliverersMap.put(udc.getDelivererName()+"."+channelId, Boolean.TRUE);
        }

        // use for debugging, uncomment for production
        //LOG.info("CurrentDeliverersMap");
        //Iterator iter = currentDeliverersMap.keySet().iterator();
        //while (iter.hasNext()) {
        //   Object o = iter.next();	   
        //   LOG.info("key: "+o.toString()+", value: "+ currentDeliverersMap.get(o) );
        //}

        model.put("channels", channels);
        model.put("deliveryTypes", deliveryTypes);
        model.put("preferences", preferences);
        model.put("currentDeliverersMap", currentDeliverersMap);
        model.put("message", "Update Successful");
        putBackLocation(model, request.getParameter("backLocation"));

        return new ModelAndView(VIEW, model);
    }

    public RecipientPreferenceService getRecipientPreferenceService() {
        return recipientPreferenceService;
    }

    public MessageDelivererRegistryService getMessageDelivererRegistryService() {
        return messageDelivererRegistryService;
    }

    public KENIntegrationService getKenIntegrationService() {
        return kenIntegrationService;
    }

    public void setRecipientPreferenceService(RecipientPreferenceService userPreferenceService) {
        this.recipientPreferenceService = userPreferenceService;
    }

    public void setMessageDelivererRegistryService(MessageDelivererRegistryService messageDelivererRegistryService) {
        this.messageDelivererRegistryService = messageDelivererRegistryService;
    }

    public void setKenIntegrationService(KENIntegrationService kis) {
        this.kenIntegrationService = kis;
    }
}
