/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.kew.rule.dao.impl;

import org.apache.commons.lang.StringUtils;
import org.kuali.rice.core.api.criteria.OrderByField;
import org.kuali.rice.core.api.criteria.OrderDirection;
import org.kuali.rice.core.api.criteria.Predicate;
import org.kuali.rice.core.api.criteria.QueryResults;
import org.kuali.rice.kew.rule.bo.RuleTemplateBo;
import org.kuali.rice.kew.rule.dao.RuleTemplateDAO;
import org.kuali.rice.krad.data.DataObjectService;
import org.kuali.rice.krad.data.PersistenceOption;
import org.kuali.rice.krad.data.platform.MaxValueIncrementerFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.jdbc.support.incrementer.DataFieldMaxValueIncrementer;

import javax.persistence.EntityManager;
import javax.sql.DataSource;
import java.util.ArrayList;
import java.util.List;

import static org.kuali.rice.core.api.criteria.PredicateFactory.equal;
import static org.kuali.rice.core.api.criteria.PredicateFactory.likeIgnoreCase;


public class RuleTemplateDAOJpa implements RuleTemplateDAO {

    private static final String SEQUENCE_NAME = "KREW_RTE_TMPL_S";

	private EntityManager entityManager;
    private DataObjectService dataObjectService;

    private DataSource dataSource;
	
    @Override
    public List<RuleTemplateBo> findAll() {
        org.kuali.rice.core.api.criteria.QueryByCriteria.Builder builder =
                org.kuali.rice.core.api.criteria.QueryByCriteria.Builder.create();

        return getDataObjectService().findMatching(RuleTemplateBo.class,builder.build()).getResults();
    }

    @Override
    public RuleTemplateBo findByRuleTemplateName(String ruleTemplateName) {
        if (StringUtils.isBlank(ruleTemplateName)) {
        	return null;
        }

        org.kuali.rice.core.api.criteria.QueryByCriteria.Builder builder =
                org.kuali.rice.core.api.criteria.QueryByCriteria.Builder.create();
        builder.setPredicates(equal("name",ruleTemplateName));
        builder.setOrderByFields(OrderByField.Builder.create("name", OrderDirection.DESCENDING).build());

        QueryResults<RuleTemplateBo> ruleTemplates = getDataObjectService().findMatching(RuleTemplateBo.class,builder.build());

        if(ruleTemplates==null || ruleTemplates.getResults() == null || ruleTemplates.getResults().isEmpty()){
        	return null;
        }
        return (RuleTemplateBo) ruleTemplates.getResults().get(0);
    }

    @Override
    public List<RuleTemplateBo> findByRuleTemplate(RuleTemplateBo ruleTemplate) {
        org.kuali.rice.core.api.criteria.QueryByCriteria.Builder builder =
                org.kuali.rice.core.api.criteria.QueryByCriteria.Builder.create();
        List<Predicate> predicates = new ArrayList<Predicate>();

        if (ruleTemplate.getName() != null) {
          predicates.add(likeIgnoreCase("name",ruleTemplate.getName()));
        }
        if (ruleTemplate.getDescription() != null) {
            predicates.add(likeIgnoreCase("description",ruleTemplate.getDescription()));
        }
        Predicate[] preds = predicates.toArray(new Predicate[predicates.size()]);
        builder.setPredicates(preds);
        QueryResults<RuleTemplateBo> results = getDataObjectService().
                    findMatching(RuleTemplateBo.class, builder.build());
        return results.getResults();
    }

    @Override
    public void delete(String ruleTemplateId) {
        getDataObjectService().delete(findByRuleTemplateId(ruleTemplateId));
    }

    @Override
    public RuleTemplateBo findByRuleTemplateId(String ruleTemplateId) {
        return getDataObjectService().find(RuleTemplateBo.class, ruleTemplateId);
     }

    @Override
    public RuleTemplateBo save(RuleTemplateBo ruleTemplate) {
    	return getDataObjectService().save(ruleTemplate, PersistenceOption.FLUSH);
    }

    @Override
    public String getNextRuleTemplateId() {
        DataFieldMaxValueIncrementer incrementer = MaxValueIncrementerFactory.getIncrementer(
                getDataSource(), SEQUENCE_NAME);
        return incrementer.nextStringValue();
    }

    public EntityManager getEntityManager() {
        return this.entityManager;
    }

    public void setEntityManager(EntityManager entityManager) {
        this.entityManager = entityManager;
    }

    public DataSource getDataSource() {
        return dataSource;
    }

    @Required
    public void setDataSource(DataSource dataSource) {
        this.dataSource = dataSource;
    }

    public DataObjectService getDataObjectService() {
        return dataObjectService;
    }

    @Required
    public void setDataObjectService(DataObjectService dataObjectService) {
        this.dataObjectService = dataObjectService;
    }
}
