/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ken.web.spring;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.ken.bo.NotificationBo;
import org.kuali.rice.ken.bo.NotificationChannelReviewerBo;
import org.kuali.rice.ken.document.kew.NotificationWorkflowDocument;
import org.kuali.rice.ken.service.NotificationMessageContentService;
import org.kuali.rice.ken.service.NotificationRecipientService;
import org.kuali.rice.ken.service.NotificationWorkflowDocumentService;
import org.kuali.rice.ken.util.NotificationConstants;
import org.kuali.rice.ken.util.Util;
import org.kuali.rice.kew.api.WorkflowDocument;
import org.kuali.rice.kew.api.WorkflowDocumentFactory;
import org.kuali.rice.kim.api.identity.Person;
import org.kuali.rice.kim.api.identity.PersonService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindException;
import org.springframework.validation.ValidationUtils;
import org.springframework.web.bind.ServletRequestBindingException;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * Implements reviewer Approve/Disapprove and initiator Acknowledge of a Notification requests
 * sent to channels configured with reviewers
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
@RequestMapping(value="/ken")
@Controller("administerNotificationRequestController")
public class AdministerNotificationRequestController {
    private static final Logger LOG = LogManager.getLogger(AdministerNotificationRequestController.class);
    /** Default command name used for binding command objects: "command" */
    public static final String DEFAULT_COMMAND_NAME = "command";

    @Autowired
    @Qualifier("messageContentService")
    protected NotificationMessageContentService messageContentService;
    @Autowired
    @Qualifier("notificationWorkflowDocumentService")
    protected NotificationWorkflowDocumentService workflowDocumentService;
    @Autowired
    @Qualifier("notificationRecipientService")
    protected NotificationRecipientService recipientService;
    @Autowired
    @Qualifier("personService")
    protected PersonService personService;

    /**
     * Command object for this controller
     */
    public static class AdministerNotificationRequestCommand {
        // incoming
        private String docId;

        // outgoing
        private WorkflowDocument document;
        private NotificationBo notification;
        private String renderedContent;
        private boolean valid = true;
        private String message;

        public String getDocId() {
            return docId;
        }
        public void setDocId(String docId) {
            this.docId = docId;
        }
        public WorkflowDocument getDocument() {
            return document;
        }
        public void setDocument(WorkflowDocument document) {
            this.document = document;
        }
        public NotificationBo getNotification() {
            return notification;
        }
        public void setNotification(NotificationBo notification) {
            this.notification = notification;
        }
        public String getRenderedContent() {
            return renderedContent;
        }
        public void setRenderedContent(String renderedContent) {
            this.renderedContent = renderedContent;
        }
        public boolean isValid() {
            return valid;
        }
        public void setValid(boolean valid) {
            this.valid = valid;
        }
        public String getMessage() {
            return message;
        }
        public void setMessage(String message) {
            this.message = message;
        }
    }

    /**
     * Parses the serialized Notification xml from the workflow document application content into a reconstituted
     * Notification BO
     * @param document the WorkflowDocument
     * @return a Notification BO reconstituted from the serialized XML form in the workflow document
     * @throws Exception if parsing fails
     */
    private NotificationBo retrieveNotificationForWorkflowDocument(WorkflowDocument document) throws Exception {
        String notificationAsXml = document.getApplicationContent();

        //parse out the application content into a Notification BO
        NotificationBo notification = messageContentService.parseSerializedNotificationXml(notificationAsXml.getBytes());

        return notification;
    }

    /**
     * View action that displays an approve/disapprove/acknowledge view
     * @param request the HttpServletRequest
     * @param command the command object bound for this Controller
     * @return a view ModelAndView
     */
    @RequestMapping("/AdministerNotificationRequest.form")
    public ModelAndView view(HttpServletRequest request, @RequestBody AdministerNotificationRequestCommand command) {
        // obtain a workflow user object first
        String initiatorId = request.getRemoteUser();

        // now construct the workflow document, which will interact with workflow
        if (command.getDocId() == null) {
            throw new RuntimeException("An invalid document ID was recieved from KEW's action list.");
        }

        //check to see which view is being passed to us from the notification list - pop up or inline
        String view = request.getParameter(NotificationConstants.NOTIFICATION_CONTROLLER_CONSTANTS.COMMAND);
        String standaloneWindow = "true";
        if(view != null && view.equals(NotificationConstants.NOTIFICATION_DETAIL_VIEWS.INLINE)) {
            standaloneWindow = "false";
        }

        WorkflowDocument document;
        Map<String, Object> model = new HashMap<String, Object>();
        // set into model whether we are dealing with a pop up or an inline window
        model.put(NotificationConstants.NOTIFICATION_CONTROLLER_CONSTANTS.STANDALONE_WINDOW, standaloneWindow);
        try {
            document = NotificationWorkflowDocument.loadNotificationDocument(initiatorId, command.getDocId());

            NotificationBo notification = retrieveNotificationForWorkflowDocument(document);

            // set up model
            command.setDocument(document);
            command.setNotification(notification);
            // render the event content according to registered XSLT stylesheet
            command.setRenderedContent(Util.transformContent(notification));

            LOG.info("notification auto remove date time: " + notification.getAutoRemoveDateTime());
            if (document.isApproved()) {
                command.setValid(false);
                command.setMessage("This notification request has been approved.");
            } else if (document.isDisapproved()) {
                command.setMessage("This notification request has been disapproved.");
            } else if (notification.getAutoRemoveDateTime() != null && notification.getAutoRemoveDateTimeValue().before(new Date(System.currentTimeMillis()))) {
                /*if (!document.stateIsCanceled()) {
                workflowDocumentService.terminateWorkflowDocument(new WorkflowDocument(new NetworkIdVO("notsys"), new Long(command.getDocId())));
                }*/
                // the autoremove date time has already passed...this notification request is null and void at this time
                boolean disapproved = document.isDisapproved();
                if (!document.isDisapproved()) {
                    List<NotificationChannelReviewerBo> reviewers = notification.getChannel().getReviewers();
                    String user = null;
                    for (NotificationChannelReviewerBo reviewer: reviewers) {
                        if ("USER".equals(reviewer.getReviewerType())) {
                            if (reviewer.getReviewerId().equals(request.getRemoteUser())) {
                                user = request.getRemoteUser();
                            }
                        } else if ("GROUP".equals(reviewer.getReviewerType())) {
                            // if it's a group
                            String[] members = recipientService.getGroupMembers(reviewer.getReviewerId());
                            for (String member: members) {
                                if (StringUtils.equals(member, request.getRemoteUser())) {
                                    user = request.getRemoteUser();
                                    break;
                                }
                            }
                        }
                    }
                    // if the current user is a reviewer, then disapprove as that user
                    if (user != null) {
	                    WorkflowDocumentFactory.loadDocument(user, command.getDocId()).disapprove("Disapproving notification request.  Auto-remove datetime has already passed.");
                        disapproved = true;
                    }
                }
                command.setValid(false);
                if (disapproved) {
                    command.setMessage("This notification request is no longer valid because the Auto-Remove date has already passed.  It has been disapproved.  Please refresh your action list.");
                } else {
                    command.setMessage("This notification request is no longer valid because the Auto-Remove date has already passed.");
                }
            }

            model.put(getCommandName(command), command);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        return new ModelAndView("ViewNotificationRequestDetails", model);
    }

    /**
     * Approve action that approves a notification request
     * @param request the HttpServletRequest
     * @param command the command object bound for this Controller
     * @return a view ModelAndView
     * @throws ServletException if an error occurs during approval
     */
    @RequestMapping(path="/AdministerNotificationRequest.form", params = "approve")
    public ModelAndView approve(HttpServletRequest request, @RequestBody AdministerNotificationRequestCommand command) throws ServletException {
        administerEventNotificationMessage(request, command, "approve");
        Map<String, Object> model = new HashMap<String, Object>();
        model.put("workflowActionTaken", "Approved");
        model.put(NotificationConstants.NOTIFICATION_CONTROLLER_CONSTANTS.STANDALONE_WINDOW, request.getParameter(NotificationConstants.NOTIFICATION_CONTROLLER_CONSTANTS.STANDALONE_WINDOW));
        return new ModelAndView("SendNotificationRequestActionTakenWindow", model);
    }

    /**
     * Disapprove action that disapproves a notification request
     * @param request the HttpServletRequest
     * @param command the command object bound for this Controller
     * @return a view ModelAndView
     * @throws ServletException if an error occurs during disapproval
     */
    @RequestMapping(path="/AdministerNotificationRequest.form", params = "disapprove")
    public ModelAndView disapprove(HttpServletRequest request, @RequestBody AdministerNotificationRequestCommand command) throws ServletException {
        administerEventNotificationMessage(request, command, "disapprove");
        Map<String, Object> model = new HashMap<String, Object>();
        model.put("workflowActionTaken", "Disapproved");
        model.put(NotificationConstants.NOTIFICATION_CONTROLLER_CONSTANTS.STANDALONE_WINDOW, request.getParameter(NotificationConstants.NOTIFICATION_CONTROLLER_CONSTANTS.STANDALONE_WINDOW));
        return new ModelAndView("SendNotificationRequestActionTakenWindow", model);
    }

    /**
     * Acknowledge action that acknowledges a notification request disapproval
     * @param request the HttpServletRequest
     * @param command the command object bound for this Controller
     * @return a view ModelAndView
     * @throws ServletException if an error occurs during acknowledgement
     */
    @RequestMapping(path="/AdministerNotificationRequest.form", params = "acknowledge")
    public ModelAndView acknowledge(HttpServletRequest request, @RequestBody AdministerNotificationRequestCommand command) throws ServletException {
        administerEventNotificationMessage(request, command, "acknowledge");
        Map<String, Object> model = new HashMap<String, Object>();
        model.put(NotificationConstants.NOTIFICATION_CONTROLLER_CONSTANTS.STANDALONE_WINDOW, request.getParameter(NotificationConstants.NOTIFICATION_CONTROLLER_CONSTANTS.STANDALONE_WINDOW));
        model.put("workflowActionTaken", "Acknowledged");
        return new ModelAndView("SendNotificationRequestActionTakenWindow", model);
    }

    /**
     * This method handles approval/disapproval/acknowledgement of the notification request
     * @param request the HttpServletRequest
     * @param command the command object bound for this Controller
     * @throws ServletException
     */
    private void administerEventNotificationMessage(HttpServletRequest request, AdministerNotificationRequestCommand command, String action) throws ServletException {
        LOG.debug("remoteUser: " + request.getRemoteUser());

        BindException bindException = new BindException(command, "command");
        ValidationUtils.rejectIfEmpty(bindException, "docId", "Document id must be specified");
        if (bindException.hasErrors()) {
            throw new ServletRequestBindingException("Document id must be specified", bindException);
        }

        // obtain a workflow user object first
        String userId = request.getRemoteUser();

        try {
            // now construct the workflow document, which will interact with workflow
            WorkflowDocument document = NotificationWorkflowDocument.loadNotificationDocument(userId, command.getDocId());

            NotificationBo notification = retrieveNotificationForWorkflowDocument(document);

            String initiatorPrincipalId = document.getInitiatorPrincipalId();
            Person initiator = getPersonService().getPerson(initiatorPrincipalId);
            String notificationBlurb =  notification.getContentType().getName() + " notification submitted by " + initiator.getName() + " for channel " + notification.getChannel().getName();
            if ("disapprove".equals(action)) {
                document.disapprove("User " + userId + " disapproving " + notificationBlurb);
            } else if ("approve".equals(action)) {
                document.approve("User " + userId + " approving " + notificationBlurb);
            } else if ("acknowledge".equals(action)) {
                document.acknowledge("User " + userId + " acknowledging " + notificationBlurb);
            }
        } catch (Exception e) {
            LOG.error("Exception occurred taking action on notification request", e);
            throw new ServletException("Exception occurred taking action on notification request", e);
        }
    }

    /**
     * Return the command name to use for the given command object.
     * <p>Default is "command".
     * @param command the command object
     * @return the command name to use
     * @see #DEFAULT_COMMAND_NAME
     */
    protected String getCommandName(Object command) {
        return DEFAULT_COMMAND_NAME;
    }

    public void setMessageContentService(
            NotificationMessageContentService notificationMessageContentService) {
        this.messageContentService = notificationMessageContentService;
    }

    public void setWorkflowDocumentService(
            NotificationWorkflowDocumentService notificationWorkflowDocumentService) {
        this.workflowDocumentService = notificationWorkflowDocumentService;
    }

    public void setRecipientService(
            NotificationRecipientService notificationRecipientService) {
        this.recipientService = notificationRecipientService;
    }

    public NotificationMessageContentService getMessageContentService() {
        return messageContentService;
    }

    public NotificationWorkflowDocumentService getWorkflowDocumentService() {
        return workflowDocumentService;
    }

    public NotificationRecipientService getRecipientService() {
        return recipientService;
    }

    public PersonService getPersonService() {
        return personService;
    }

    public void setPersonService(PersonService personService) {
        this.personService = personService;
    }
}
