/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2026 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.ken.util;

import java.io.IOException;
import java.io.InputStream;

import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Internal notification EntityResolver which resolves system ids with the "resource:" prefix to ClassLoader resources
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public class ClassLoaderEntityResolver extends ClassLoaderResourceResolver implements EntityResolver {

    public ClassLoaderEntityResolver() {
        super();
    }

    public ClassLoaderEntityResolver(String base, String prefix) {
        super(base, prefix);
    }

    @Override
    public InputSource resolveEntity(String publicId, String systemId) throws SAXException, IOException {
        LOG.debug("Resolving '" + publicId + "' / '" + systemId + "'");
        String path = resolveSystemId(systemId);
        if (path == null) {
            LOG.error("Unable to resolve system id '" + systemId + "' locally...delegating to default resolution strategy.");
            return null;
        }
        LOG.debug("Looking up resource '" + path + "' for entity '" + systemId + "'");
        InputStream is = getClass().getClassLoader().getResourceAsStream(path);
        if (is == null) {
            String message = "Unable to find schema (" + path + ") for: " + systemId;
            LOG.warn(message);
            // necessary if processContents is lax, because lax doesn't care...if it doesn't resolve it won't validate
            // (not quite clear, as lax could be interpreted as *if the namespace is valid*, treating a present, but invalid
            // namespace as a fatal error. instead, apparently a present but invalid namespace is ignored with 'lax'
            // so w should use strict to ensure that is an error instead of throwing an exception here gratuitously
            // which will screw up compound resolution
            //throw new SAXException(message);
            return null;
        }
        return new InputSource(is);
    }

    @Override
    public String toString() {
        return "[ClassLoaderEntityResolver: base=" + base + ", prefix=" + prefix + "]";
    }
}
