/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2024 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.bo;

import org.kuali.rice.krad.exception.ExportNotSupportedException;

import java.io.IOException;
import java.io.OutputStream;
import java.util.List;

/**
 * An Exporter provides the ability to export a list of data objects to a supported export format.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
public interface Exporter {

    /**
     * Exports the List of BusinessObjects to the specified ExportFormat.
     *
     * <p>The resulting output of the export operation should be written to the given
     * OutputStream</p>
     *
     * @param dataObjectClass the type of DataObjects being exported
     * @param dataObjects a List of DataObjects to export
     * @param exportFormat the export format in which to export the DataObjects
     * @param outputStream the OutputStream to write the exported data to
     * @throws IOException if the process encounters an I/O issue
     * @throws ExportNotSupportedException if the given ExportFormat is not supported
     */
    void export(Class<?> dataObjectClass, List<? extends Object> dataObjects, String exportFormat,
            OutputStream outputStream) throws IOException, ExportNotSupportedException;

    /**
     * Returns a List of ExportFormats supported by this Exporter for the given DataOject class.
     *
     * @param dataObjectClass the class of the DataObjects being exported
     * @return a List of supported ExportFormats
     */
    List<String> getSupportedFormats(Class<?> dataObjectClass);

}
