/*-
 * #%L
 * %%
 * Copyright (C) 2005 - 2025 Kuali, Inc. - All Rights Reserved
 * %%
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 * 
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 * #L%
 */

package org.kuali.rice.krad.data.jpa;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.Query;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.*;
import javax.persistence.criteria.CriteriaQuery;

import org.apache.commons.lang.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.kuali.rice.core.api.criteria.PropertyPath;
import org.kuali.rice.core.api.criteria.QueryByCriteria;

/**
 * JPA QueryTranslator that translates queries directly into native JPA 2 Criteria API.
 *
 * @author Kuali Rice Team (rice.collab@kuali.org)
 */
class NativeJpaQueryTranslator extends QueryTranslatorBase<TranslationContext, TypedQuery<?>, TypedQuery<Long>> {

    /**
     * Wildcard characters that are allowed in queries.
     */
	protected static final String[] LOOKUP_WILDCARDS = { "*", "?" };

    /**
     * Wildcard characters that are allowed in queries (in their escape formats).
     */
	protected static final String[] ESCAPED_LOOKUP_WILDCARDS = { "\\*", "\\?" };

    /**
     * Wildcard character equivalents in JPQL.
     */
	protected static final char[] JPQL_WILDCARDS = { '%', '_' };

    /**
     * The entity manager for interacting with the database.
     */
    protected EntityManager entityManager;

    /**
     * Creates a native JPA translator for queries.
     *
     * @param entityManager the entity manager to use for interacting with the database.
     */
    public NativeJpaQueryTranslator(EntityManager entityManager) {
        this.entityManager = entityManager;
    }

    @Override
    public TypedQuery<?> createQuery(Class<?> queryClazz, TranslationContext criteria) {
        CriteriaQuery<?> jpaQuery = criteria.query;
        // it is important to not create an empty or predicate
        if (!criteria.predicates.isEmpty()) {
            jpaQuery = jpaQuery.where(criteria.getCriteriaPredicate());
        }
        return entityManager.createQuery(jpaQuery);
    }

    @Override
    public TypedQuery<Long> createQueryForCount(Class<?> queryClazz, TranslationContext criteriaCount) {
        CriteriaQuery<Long> jpaQuery = (CriteriaQuery<Long>) criteriaCount.query;
        // it is important to not create an empty or predicate
        if (!criteriaCount.predicates.isEmpty()) {
            jpaQuery = jpaQuery.where(criteriaCount.getCriteriaPredicate());
        }
        return entityManager.createQuery(jpaQuery);
    }

    @Override
	public Query createDeletionQuery(Class<?> queryClazz, TranslationContext criteria) {
        CriteriaDelete<?> jpaQuery = entityManager.getCriteriaBuilder().createCriteriaDelete(queryClazz);

        if (!criteria.predicates.isEmpty()) {
            jpaQuery = jpaQuery.where(criteria.getCriteriaPredicate());
        }

        return entityManager.createQuery(jpaQuery);
    }

    @Override
    protected TranslationContext createCriteria(Class<?> queryClazz) {
        return TranslationContext.createCriteria(entityManager, queryClazz);
    }

    @Override
    protected TranslationContext createCriteriaForCount(Class<?> queryClazz) {
        return TranslationContext.createCriteriaForCount(entityManager, queryClazz);
    }

    @Override
	protected TranslationContext createCriteriaForSubQuery(Class<?> queryClazz, TranslationContext parentContext) {
		return TranslationContext.createCriteriaForSubQuery(entityManager, queryClazz, parentContext);
	}


	@Override
    protected TranslationContext createInnerCriteria(TranslationContext parent) {
        // just a container for the inner predicates
        // copy everything else
        return TranslationContext.createInnerCriteria(parent);
    }

    @Override
    public void convertQueryFlags(QueryByCriteria qbc, TypedQuery<?> query) {
        final int startAtIndex = qbc.getStartAtIndex() != null ? qbc.getStartAtIndex() : 0;

        query.setFirstResult(startAtIndex);

        if (qbc.getMaxResults() != null) {
            //not subtracting one from MaxResults in order to retrieve
            //one extra row so that the MoreResultsAvailable field can be set
            query.setMaxResults(qbc.getMaxResults());
        }
    }

    @Override
    protected void addAnd(TranslationContext criteria, TranslationContext inner) {
        criteria.and(inner);
    }

    @Override
    protected void addNotNull(TranslationContext criteria, String propertyPath) {
        criteria.addPredicate(criteria.builder.isNotNull(criteria.attr(propertyPath)));
    }

    @Override
    protected void addIsNull(TranslationContext criteria, String propertyPath) {
        criteria.addPredicate(criteria.builder.isNull(criteria.attr(propertyPath)));
    }

	/**
	 * Translates the Rice Criteria API {@link PropertyPath} object into a native JPA path which can be used in JPA
	 * predicates.
	 * 
	 * @param criteria
	 *            The base criteria context for translation of the property if no specific data type is given.
	 * @param value
	 *            The {@link PropertyPath} object passed in from the Rice Criteria API.
	 * @return A JPA {@link Path} object which can be used in JPA {@link Predicate} statements.
	 */
	protected Expression<?> translatePropertyPathIntoJpaPath(TranslationContext criteria, PropertyPath value) {
		TranslationContext tempCriteria = criteria;
		if (value.getDataType() != null) {
			try {
				tempCriteria = createCriteria(Class.forName(value.getDataType()));
			} catch (ClassNotFoundException e) {
				// unable to find the type - ignore and attempt to resolve path without special context
				LogManager.getLogger(this.getClass()).error(
						"Unable to find data type " + value.getDataType()
								+ ".  Falling back to the base root for the query: " + criteria.root.getJavaType(), e);
			}
		}
		return tempCriteria.attr(value.getPropertyPath());
	}

    @Override
    protected void addEqualTo(TranslationContext criteria, String propertyPath, Object value) {
		// If this is a property path criteria, we need to translate it first
		if (value instanceof PropertyPath) {
			// We *must* make the call separate here. If we don't, it binds to the (Expression,Object) version of the
			// JPA method
			// which converts our property path into a string literal.
			Expression<?> path = translatePropertyPathIntoJpaPath(criteria, (PropertyPath) value);
			criteria.addPredicate(criteria.builder.equal(criteria.attr(propertyPath), path));
		} else {
			criteria.addPredicate(criteria.builder.equal(criteria.attr(propertyPath), value));
		}
    }

    @Override
    protected void addEqualToIgnoreCase(TranslationContext criteria, String propertyPath, String value) {
        criteria.addPredicate(criteria.builder.equal(criteria.builder.upper(criteria.attr(propertyPath)), value.toUpperCase()));
    }

    @Override
    protected void addGreaterOrEqualTo(TranslationContext criteria, String propertyPath, Object value) {
        criteria.addPredicate(criteria.builder.greaterThanOrEqualTo(criteria.attr(propertyPath), (Comparable) value));
    }

    @Override
    protected void addGreaterThan(TranslationContext criteria, String propertyPath, Object value) {
        criteria.addPredicate(criteria.builder.greaterThan(criteria.attr(propertyPath), (Comparable) value));
    }

    @Override
    protected void addLessOrEqualTo(TranslationContext criteria, String propertyPath, Object value) {
        criteria.addPredicate(criteria.builder.lessThanOrEqualTo(criteria.attr(propertyPath), (Comparable) value));
    }

    @Override
    protected void addLessThan(TranslationContext criteria, String propertyPath, Object value) {
        criteria.addPredicate(criteria.builder.lessThan(criteria.attr(propertyPath), (Comparable) value));
    }

    @Override
    protected void addLike(TranslationContext criteria, String propertyPath, Object value) {
        // value should be a String pattern
		criteria.addPredicate(criteria.builder.like(criteria.attr(propertyPath), fixSearchPattern(value.toString())));
    }

    @Override
    protected void addLikeIgnoreCase(TranslationContext criteria, String propertyPath, String value){
        criteria.addPredicate(criteria.builder.like(criteria.builder.upper(criteria.attr(propertyPath)),
                fixSearchPattern(value.toUpperCase())));
    }

	@Override
	protected void addNotLikeIgnoreCase(TranslationContext criteria, String propertyPath, String value) {
		criteria.addPredicate(criteria.builder.notLike(criteria.builder.upper(criteria.attr(propertyPath)),
				fixSearchPattern(value.toUpperCase())));
	}

	@Override
	protected void addExistsSubquery(TranslationContext criteria, String subQueryType,
			org.kuali.rice.core.api.criteria.Predicate subQueryPredicate) {
		try {
			Class<?> subQueryBaseClass = Class.forName(subQueryType);
			Subquery<?> subquery = criteria.query.subquery(subQueryBaseClass);
			TranslationContext subQueryJpaPredicate = createCriteriaForSubQuery(subQueryBaseClass, criteria);

			// If a subQueryPredicate is passed, this is a Rice Predicate object and must be translated
			// into JPA - so we add it to the list this way.
			if (subQueryPredicate != null) {
				addPredicate(subQueryPredicate, subQueryJpaPredicate);
			}

			subquery.where(subQueryJpaPredicate.predicates.toArray(new Predicate[0]));
			criteria.addExistsSubquery(subquery);
		} catch (ClassNotFoundException e) {
			throw new IllegalArgumentException(subQueryType + " can not be resolved to a class for JPA", e);
		}
	}

	/**
	 * Fixes the search pattern by converting all non-escaped lookup wildcards ("*" and "?") into their respective JPQL
	 * wildcards ("%" and "_").
     *
     * <p>Any lookup wildcards escaped by a backslash are converted into their non-backslashed equivalents.</p>
     *
     * @param value the search pattern to fix.
     * @return a fixed search pattern.
	 */
	protected String fixSearchPattern(String value) {
		StringBuilder fixedPattern = new StringBuilder(value);
		// Convert all non-escaped wildcards.
		for (int i = 0; i < LOOKUP_WILDCARDS.length; i++) {
			String lookupWildcard = LOOKUP_WILDCARDS[i];
			String escapedLookupWildcard = ESCAPED_LOOKUP_WILDCARDS[i];
			char jpqlWildcard = JPQL_WILDCARDS[i];
			int wildcardIndex = fixedPattern.indexOf(lookupWildcard);
			int escapedWildcardIndex = fixedPattern.indexOf(escapedLookupWildcard);
			while (wildcardIndex != -1) {
				if (wildcardIndex == 0 || escapedWildcardIndex != wildcardIndex - 1) {
					fixedPattern.setCharAt(wildcardIndex, jpqlWildcard);
					wildcardIndex = fixedPattern.indexOf(lookupWildcard, wildcardIndex);
				} else {
					fixedPattern.replace(escapedWildcardIndex, wildcardIndex + 1, lookupWildcard);
					wildcardIndex = fixedPattern.indexOf(lookupWildcard, wildcardIndex);
					escapedWildcardIndex = fixedPattern.indexOf(escapedLookupWildcard, wildcardIndex);
				}
			}
		}
		return fixedPattern.toString();
	}

    @Override
    protected void addNotEqualTo(TranslationContext criteria, String propertyPath, Object value) {
		// If this is a property path criteria, we need to translate it first
		if (value instanceof PropertyPath) {
			// We *must* make the call separate here. If we don't, it binds to the (Expression,Object) version of the
			// JPA method
			// which converts our property path into a string literal.
			Expression<?> path = translatePropertyPathIntoJpaPath(criteria, (PropertyPath) value);
			criteria.addPredicate(criteria.builder.notEqual(criteria.attr(propertyPath), path));
		} else {
			criteria.addPredicate(criteria.builder.notEqual(criteria.attr(propertyPath), value));
		}
    }

    @Override
    protected void addNotEqualToIgnoreCase(TranslationContext criteria, String propertyPath, String value) {
        criteria.addPredicate(criteria.builder.notEqual(criteria.builder.upper(criteria.attr(propertyPath)), value.toUpperCase()));
    }

    @Override
    protected void addNotLike(TranslationContext criteria, String propertyPath, Object value) {
        // value should be a String pattern
		criteria.addPredicate(criteria.builder.notLike(criteria.attr(propertyPath), fixSearchPattern(value.toString())));
    }

    @Override
    protected void addIn(TranslationContext criteria, String propertyPath, Collection<?> values) {
        criteria.addPredicate(criteria.attr(propertyPath).in(values));
    }

    @Override
    protected void addNotIn(TranslationContext criteria, String propertyPath, Collection<?> values) {
        criteria.addPredicate(criteria.builder.not(criteria.attr(propertyPath).in(values)));
    }

    @Override
    protected void addOr(TranslationContext criteria, TranslationContext inner) {
        criteria.or(inner);
    }

    @Override
    protected void addOrderBy(TranslationContext criteria, String propertyPath, boolean sortAscending) {
        List<Order> orderList = criteria.query.getOrderList();
        if (orderList == null) {
            orderList = new ArrayList<>();
        }

        if (propertyPath.contains(".")) {
            String propertyPathStart = StringUtils.substringBefore( propertyPath, "." );
            String propertyPathEnd = StringUtils.substringAfter( propertyPath, "." );

            if (sortAscending) {
                orderList.add(criteria.builder.asc(criteria.root.get(propertyPathStart).get(propertyPathEnd)));
            } else {
                orderList.add(criteria.builder.desc(criteria.root.get(propertyPathStart).get(propertyPathEnd)));
            }
        } else {
            if (sortAscending) {
                orderList.add(criteria.builder.asc(criteria.root.get(propertyPath)));
            } else {
                orderList.add(criteria.builder.desc(criteria.root.get(propertyPath)));
            }
        }

        criteria.query.orderBy(orderList);
    }

    @Override
    protected String genUpperFunc(String pp) {
        throw new IllegalStateException("genUpperFunc should not have been invoked for NativeJpaQueryTranslator");
    }

}
